<?php

declare(strict_types=1);

namespace Drupal\Tests\babel\FunctionalJavascript;

use Behat\Mink\Element\NodeElement;
use Drupal\Core\Url;
use Drupal\user\Entity\Role;
use Drupal\user\RoleInterface;

/**
 * Tests related to the Toolbar and the translate modal form.
 */
class TranslationToolbarModalTest extends BabelFunctionalJavascriptTestBase {

  /**
   * CSS selector of the "Translate" toolbar link.
   *
   * @const string
   */
  protected const TRANSLATE_TOOLBAR_CSS_SELECTOR = '.toolbar-tab.toolbar-tab--babel-translate a';

  /**
   * {@inheritdoc}
   */
  protected const TEST_USER_PERMISSIONS = [
    'access toolbar',
    'translate interface',
  ];

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'toolbar',
  ];

  /**
   * {@inheritdoc}
   */
  public function setUp(): void {
    parent::setUp();

    Role::load(RoleInterface::ANONYMOUS_ID)
      ->grantPermission('access toolbar')
      ->save();
  }

  /**
   * Tests the visibility of the "Translate" toolbar link.
   */
  public function testToolbarLink(): void {
    $this->drupalGet('<front>');
    // Toolbar must exist.
    $this->assertToolbarPresent();
    // ...But no "Translate" link must be present.
    $this->assertNoBabelTranslateToolbarLinkPresent();

    $this->drupalLogin($this->testAccount);
    $this->drupalGet('<front>');
    // Toolbar and also the Translate toolbar item must exist.
    $this->assertToolbarPresent();
    $this->assertBabelTranslateToolbarLinkPresent();

    // On the Babel translate UI route, the Toolbar link should not be visible.
    $this->drupalGet(Url::fromRoute('babel.ui'));
    $this->assertToolbarPresent();
    $this->assertNoBabelTranslateToolbarLinkPresent();
  }

  /**
   * Tests Babel's translation UI opened into a modal dialog.
   */
  public function testTranslationUiModal(): void {
    $this->drupalLogin($this->testAccount);
    $this->drupalGet('<front>');
    $this->assertBabelTranslateToolbarLinkPresent()->click();
    $this->assertSession()->waitForElement('css', '.ui-dialog--babel-translate');

    $this->assertTranslationForm();
  }

  /**
   * Verifies that the Toolbar is present.
   */
  protected function assertToolbarPresent(): void {
    $this->assertInstanceOf(
      NodeElement::class,
      $this->assertSession()->waitForElement('css', 'nav.toolbar-bar'),
    );
  }

  /**
   * Verifies that the "Translate" toolbar link is available.
   */
  protected function assertBabelTranslateToolbarLinkPresent(): NodeElement {
    $toolbarLink = $this->assertSession()->waitForElement('css', static::TRANSLATE_TOOLBAR_CSS_SELECTOR);
    $this->assertInstanceOf(NodeElement::class, $toolbarLink);

    return $toolbarLink;
  }

  /**
   * Verifies that the "Translate" toolbar link is not present.
   */
  protected function assertNoBabelTranslateToolbarLinkPresent(): void {
    $this->assertToolbarPresent();
    $this->assertSession()->assertNoElementAfterWait('css', static::TRANSLATE_TOOLBAR_CSS_SELECTOR);
  }

}
