<?php

declare(strict_types=1);

namespace Drupal\babel_content_entity\Form;

use Drupal\babel\BabelStorageInterface;
use Drupal\babel_content_entity\BabelContentEntityService;
use Drupal\Component\Render\MarkupInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\DependencyInjection\AutowireTrait;
use Drupal\Core\Entity\ContentEntityTypeInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\ConfigTarget;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;

/**
 * Babel Content Entity settings form.
 */
class BabelContentEntitySettingsForm extends ConfigFormBase {

  use AutowireTrait;

  public function __construct(
    ConfigFactoryInterface $config_factory,
    TypedConfigManagerInterface $typedConfigManager,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected BabelStorageInterface $babelStorage,
    protected BabelContentEntityService $service,
    protected EntityFieldManagerInterface $fieldManager,
    protected EntityTypeBundleInfoInterface $bundleInfo,
  ) {
    parent::__construct($config_factory, $typedConfigManager);
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildForm($form, $form_state);

    $form_state->set('entity_type', $this->config('babel_content_entity.settings')->get('entity_type'));

    $form['entity_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Entity types'),
      '#options' => $this->getContentEntityOptions(),
      '#multiple' => TRUE,
      '#size' => 12,
      '#config_target' => new ConfigTarget(
        configName: 'babel_content_entity.settings',
        propertyPath: 'entity_type',
        toConfig: [static::class, 'normalize'],
      ),
      '#description' => [
        [
          '#markup' => $this->t('IMPORTANT: Beware that selecting some entity types, with a lot of entities, could bring a huge amount of strings with a high impact on performance.'),
        ],
        [
          '#prefix' => ' ',
          '#markup' => $this->t('Limit your selection to entity types with a predictable and controlled amount of entities.'),
        ],
      ],
    ];
    return $form;
  }

  /**
   * Returns a list of eligible entity types to be used as form API options.
   *
   * @return array<non-empty-string, MarkupInterface|string>
   *   List of eligible entity types to be used as form API options.
   */
  protected function getContentEntityOptions(): array {
    $options = [];
    foreach ($this->entityTypeManager->getDefinitions() as $entityTypeId => $entityType) {
      if (!$entityType instanceof ContentEntityTypeInterface || !$entityType->isTranslatable()) {
        continue;
      }

      foreach ($this->bundleInfo->getBundleInfo($entityTypeId) as $bundle => $bundleInfo) {
        if ($this->service->getBundleTranslatableStringFields($entityTypeId, $bundle)) {
          $options[$entityTypeId] = $entityType->getLabel();
          // Jump to next entity type.
          break;
        }
      }
    }

    // Temporarily, the menu_link_content entity type is handled by the
    // babel_menu_link_content module. In https://www.drupal.org/i/3547227, it
    // will be converted to use this plugin and babel_menu_link_content module
    // will be deprecated.
    // @see https://www.drupal.org/i/3547227
    unset($options['menu_link_content']);

    // Sort by label.
    asort($options);

    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    parent::submitForm($form, $form_state);

    $initial = $form_state->get('entity_type');
    $final = $this->config('babel_content_entity.settings')->get('entity_type');

    if ($removed = array_diff($initial, $final)) {
      $this->messenger()->addStatus($this->formatPlural(
        count($removed),
        'Strings belonging to %types entity type are no more translatable.',
        'Strings belonging to %types entity types are no more translatable.',
        ['%types' => implode(', ', $this->getAffectedEntityTypeLabels($removed))]
      ));
    }

    if ($added = array_diff($final, $initial)) {
      $this->messenger()->addStatus([
        [
          '#markup' => $this->formatPlural(
            count($added),
            'Strings belonging to %types entity are now exposed for translation.',
            'Strings belonging to %types entity types are now exposed for translation.',
            ['%types' => implode(', ', $this->getAffectedEntityTypeLabels($added))],
          ),
        ],
        [
          '#prefix' => ' ',
          '#markup' => $this->t('It is strongly recommended to visit the <a href=":url">settings page</a> to narrow down the bundles that need translation for each entity type.', [
            ':url' => Url::fromRoute('babel.settings')->toString(),
          ]),
        ],
      ]);
    }
  }

  /**
   * Normalizes the form entity_type value.
   *
   * @param array<non-empty-string, non-empty-string> $value
   *   Submitted value.
   *
   * @return list<non-empty-string>
   *   Config stored value.
   */
  public static function normalize(array $value): array {
    return array_keys(array_filter($value));
  }

  /**
   * Returns a list of affected entity type labels.
   *
   * @param array $entityTypeIds
   *   Entity type IDs.
   *
   * @return array<non-empty-string, MarkupInterface|string>
   *   A list of affected entity type labels.
   */
  protected function getAffectedEntityTypeLabels(array $entityTypeIds): array {
    return array_map(
      fn(ContentEntityTypeInterface $entityType): string|MarkupInterface => $entityType->getLabel(),
      array_intersect_key($this->entityTypeManager->getDefinitions(), array_flip($entityTypeIds)),
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['babel_content_entity.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'babel_content_entity_settings_form';
  }

}
