<?php

declare(strict_types=1);

namespace Drupal\Tests\babel_content_entity\Kernel\Plugin\Babel\TranslationType;

use Drupal\babel\Model\Source;
use Drupal\babel\Model\StringTranslation;
use Drupal\babel\Plugin\Babel\TranslationTypePluginManager;
use Drupal\babel_content_entity\Plugin\Babel\TranslationType\ContentEntity;
use Drupal\taxonomy\Entity\Vocabulary;
use Drupal\taxonomy\TermInterface;
use Drupal\Tests\babel\Kernel\Plugin\Babel\TranslationType\TranslationTypePluginTestBase;

/**
 * Tests the 'content_entity' translation type plugin.
 *
 * @group babel
 *
 * @coversDefaultClass \Drupal\babel_content_entity\Plugin\Babel\TranslationType\ContentEntity
 */
class ContentEntityTest extends TranslationTypePluginTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'content_translation',
    'filter',
    'taxonomy',
    'text',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('taxonomy_term');

    $this->container->get('module_installer')->install(['babel_content_entity']);
    $this->config('babel_content_entity.settings')
      ->set('entity_type', ['taxonomy_term'])
      ->save();

    $this->plugin = $this->container
      ->get(TranslationTypePluginManager::class)
      ->createInstance(ContentEntity::id('taxonomy_term'));

    Vocabulary::create(['vid' => 'tags', 'name' => 'Tags'])->save();

    $this->container->get('content_translation.manager')
      ->setEnabled('taxonomy_term', 'tags', TRUE);
  }

  /**
   * {@inheritdoc}
   *
   * @param int $termsCount
   *   Number of the preexisting terms.
   * @param string $langcode
   *   The language code to test the loading with.
   * @param array $stringIds
   *   The string IDs to load.
   * @param array $expected
   *   The expected result.
   *
   * @dataProvider providerTestGetMultiple
   */
  public function testGetStrings(
    int $termsCount = 0,
    string $langcode = '',
    array $stringIds = [],
    array $expected = [],
  ): void {
    if ($termsCount) {
      while ($termsCount > 0) {
        $this->createRandomTerm();
        $termsCount--;
      }
    }
    $this->assertEquals(
      $expected,
      array_keys($this->plugin->getStrings($langcode, $stringIds))
    );
  }

  /**
   * Test data provider for ::testGetStrings.
   *
   * @return array[]
   *   The test cases.
   */
  public static function providerTestGetMultiple(): array {
    return [
      'Load everything, no term' => [
        'termsCount' => 0,
        'langcode' => 'fi',
        'stringIds' => [],
        'expected' => [],
      ],
      'Load everything, some links' => [
        'termsCount' => 1,
        'langcode' => 'fi',
        'stringIds' => [],
        'expected' => ['tags:1:name:0'],
      ],
      'Load specific' => [
        'termsCount' => 4,
        'langcode' => 'be',
        'stringIds' => [
          'tags:2:name:0',
          'tags:4:name:0',
        ],
        'expected' => [
          'tags:2:name:0',
          'tags:4:name:0',
        ],
      ],
    ];
  }

  /**
   * {@inheritdoc}
   *
   * @param int $termsCount
   *   Number of the preexisting terms.
   * @param string $langcode
   *   The language code to test the loading with.
   * @param array $stringId
   *   The string ID to load.
   * @param \Drupal\babel\Model\StringTranslation|null $expected
   *   The expected result.
   *
   * @dataProvider providerTestGetSingle
   */
  public function testGetString(
    int $termsCount = 0,
    string $langcode = '',
    string $stringId = '',
    ?StringTranslation $expected = NULL,
  ): void {
    if ($termsCount > 0) {
      while ($termsCount > 0) {
        $this->createRandomTerm();
        $termsCount--;
      }
    }

    $this->assertEquals(
      $expected,
      $this->plugin->getString($langcode, $stringId)
    );
  }

  /**
   * Test data provider for ::testGetString.
   *
   * @return array[]
   *   The test cases.
   */
  public static function providerTestGetSingle(): array {
    return [
      'Load specific ID' => [
        'termsCount' => 2,
        'langcode' => 'fi',
        'stringId' => 'tags:2:name:0',
        'expected' => new StringTranslation(
          source: new Source('Tag 2', '', TRUE),
        ),
      ],
      'Missing ID' => [
        'termsCount' => 0,
        'langcode' => 'fi',
        'stringId' => 'tags:2:name:0',
        'expected' => NULL,
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function testUpdateTranslation(): void {
    $term = $this->createRandomTerm('Test term name', 'Some description');

    $this->assertTrue($term->isTranslatable());
    $this->assertFalse($term->hasTranslation($this->language->getId()));

    // Add translation.
    $this->plugin->updateTranslation(
      $this->plugin->getString(
        $this->language->getId(),
        'tags:1:name:0',
      ),
      'tags:1:name:0',
      $this->language->getId(),
      '[TRANSLATED] Test term name',
    );
    $this->plugin->updateTranslation(
      $this->plugin->getString(
        $this->language->getId(),
        'tags:1:description:0',
      ),
      'tags:1:description:0',
      $this->language->getId(),
      '[TRANSLATED] Some description',
    );

    $term = \Drupal::entityTypeManager()->getStorage('taxonomy_term')->load($term->id());

    $this->assertTrue($term->hasTranslation($this->language->getId()));
    $this->assertEquals('[TRANSLATED] Test term name', $term->getTranslation($this->language->getId())->label());
    $this->assertEquals('[TRANSLATED] Some description', $term->getTranslation($this->language->getId())->getDescription());
  }

  /**
   * Creates a random taxonomy term.
   *
   * @param string|null $name
   *   (optional) The term name. Defaults to "Tag <TID>".
   * @param string|null $description
   *   (optional) The term description.
   */
  protected function createRandomTerm(?string $name = NULL, ?string $description = NULL): TermInterface {
    $storage = \Drupal::entityTypeManager()->getStorage('taxonomy_term');
    $nextId = 1;
    $highestIdResult = $storage->getQuery()->accessCheck(FALSE)->sort('tid', 'DESC')->range(0, 1)->execute();
    $nextId += current($highestIdResult) ?: 0;

    $term = $storage->create(array_filter([
      'vid' => 'tags',
      'name' => $name ?: "Tag $nextId",
      'langcode' => 'en',
      'description' => $description,
    ]));
    $term->save();

    return $term;
  }

}
