<?php

declare(strict_types=1);

namespace Drupal\babel;

use Drupal\babel\Model\Source;
use Drupal\Component\Utility\NestedArray;

/**
 * Handles batch processing of translatable strings during Babel installation.
 */
class BabelBatchProcessor {

  /**
   * Imports all existing locale source strings into Babel during installation.
   *
   * @param array $context
   *   The batch operation context, updated with progress information.
   */
  public static function importExistingLocaleStrings(array &$context): void {
    // Populate with locale source strings.
    $rows = \Drupal::database()->select('locales_source', 'ls')
      ->fields('ls', ['lid', 'source', 'context'])
      ->execute()
      ->fetchAll(\PDO::FETCH_NUM);

    $sources = [];
    foreach ($rows as [$lid, $source, $string_context]) {
      $sources[$lid] = new Source(string: $source, context: $string_context);
    }
    \Drupal::service(BabelStorageInterface::class)->update('locale', $sources);

    $context['message'] = t('Processed @count locale source strings.', ['@count' => count($sources)]);
  }

  /**
   * Processes translatable configuration strings during installation.
   *
   * @param string[] $names
   *   Array of configuration object names to process in this batch chunk.
   * @param int $total
   *   Total number of configuration objects being processed across all batches.
   * @param array $context
   *   The batch operation context, tracks progress and displays messages.
   */
  public static function processConfigurationStrings(array $names, int $total, array &$context): void {
    $context['results']['progress'] ??= 0;

    $sources = [];
    foreach ($names as $name) {
      $data = \Drupal::service('config.storage')->read($name);

      $paths = \Drupal::service(BabelConfigHelperInterface::class)->getTranslatableProperties($name);
      foreach ($paths as $path => $translationContext) {
        $sources["$name:$path"] = new Source(
          string: NestedArray::getValue($data, explode('.', $path)),
          context: $translationContext,
        );
      }
      $context['results']['progress']++;
    }

    \Drupal::service(BabelStorageInterface::class)->update('config', $sources);

    $context['message'] = t('Processed @progress out of @total configurations', [
      '@progress' => $context['results']['progress'],
      '@total' => $total,
    ]);
  }

}
