<?php

declare(strict_types=1);

namespace Drupal\babel;

use Drupal\babel\Model\StringTranslation;
use Drupal\babel\Plugin\Babel\TranslationTypePluginManager;
use Drupal\Core\Database\Connection;

/**
 * Repository service for strings.
 */
class BabelStringsRepository implements BabelStringsRepositoryInterface {

  public function __construct(
    protected readonly Connection $db,
    protected readonly TranslationTypePluginManager $manager,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function getStrings(
    string $langcode,
    ?bool $translationStatus = NULL,
    ?string $search = NULL,
    ?bool $sourceStatus = NULL,
  ): array {
    $query = $this->db->select('babel_source_instance', 'bsi')
      ->fields('bsi', ['hash', 'plugin', 'id'])
      ->fields('bs', ['status'])
      ->orderBy('bs.sort_key');
    $query->innerJoin('babel_source', 'bs', 'bsi.hash = bs.hash');

    // A specific status is requested by the caller.
    if (is_bool($sourceStatus)) {
      $query->condition('bs.status', $sourceStatus);
    }

    $data = [];
    foreach ($query->execute() as $row) {
      $data[$row->hash]['plugin'][$row->plugin][] = $row->id;
      $data[$row->hash]['status'] = $row->status;
    }

    $strings = [];
    foreach ($data as $hash => ['plugin' => $byPlugin, 'status' => $status]) {
      foreach ($byPlugin as $pluginId => $ids) {
        $plugin = $this->manager->createInstance($pluginId);

        foreach ($plugin->getStrings($langcode, $ids) as $string) {
          if (!$this->getTranslationConditions($string, $translationStatus, $strings)) {
            continue;
          }
          if (!$this->getSearchCondition($string, $search)) {
            continue;
          }

          $string->source->setStatus((bool) $status);
          $strings[$hash] = $string;
        }
      }
    }

    return $strings;
  }

  /**
   * Checks if the 'translated string' condition is satisfied.
   *
   * @param \Drupal\babel\Model\StringTranslation $string
   *   The string object (translated or not).
   * @param bool|null $translationStatus
   *   The translation status to check:
   *   - TRUE: Only translated strings.
   *   - FALSE: Only untranslated strings.
   *   - NULL: Both translated and untranslated strings.
   * @param array<array-key, \Drupal\babel\Model\StringTranslation> $strings
   *   List of already collected strings.
   *
   * @return bool
   *   Whether the 'translated string' condition is satisfied.
   */
  protected function getTranslationConditions(StringTranslation $string, ?bool $translationStatus, array $strings): bool {
    return match(TRUE) {
      // Add only if it's not yet in the list or if it's translated.
      $translationStatus === NULL => !isset($strings[$string->source->getHash()]) || $string->isTranslated(),
      $translationStatus => $string->isTranslated(),
      !$translationStatus => !$string->isTranslated(),
    };
  }

  /**
   * Checks if the search for string condition is satisfied.
   *
   * @param \Drupal\babel\Model\StringTranslation $string
   *   The string object (translated or not).
   * @param string|null $search
   *   The string contained in source or translation to filter on. If NULL is
   *   passed no filtering occurs.
   *
   * @return bool
   *   Whether the search for string condition is satisfied.
   */
  protected function getSearchCondition(StringTranslation $string, ?string $search): bool {
    if ($search === NULL || !($search = trim($search))) {
      // Condition satisfied: No valid search string has been provided.
      return TRUE;
    }

    $search = mb_strtolower($search);
    $sourceContains = str_contains(mb_strtolower($string->source->string), $search);
    $translationContains = $string->isTranslated() && str_contains(mb_strtolower($string->getTranslation()->string), $search);

    return $sourceContains || $translationContains;
  }

}
