<?php

namespace Drupal\babel\Form;

use Drupal\babel\BabelStorageInterface;
use Drupal\babel\Plugin\Babel\DataTransferPluginManager;
use Drupal\babel\Plugin\Babel\TranslationTypePluginManager;
use Drupal\Component\Gettext\PoItem;
use Drupal\Component\Utility\Html;
use Drupal\Core\Ajax\AjaxHelperTrait;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\ReplaceCommand;
use Drupal\Core\Batch\BatchBuilder;
use Drupal\Core\DependencyInjection\AutowireTrait;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\StringTranslation\PluralTranslatableMarkup;
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * Import form.
 */
class BabelImportForm extends FormBase {

  use AjaxHelperTrait;
  use AutowireTrait;

  public function __construct(
    protected DataTransferPluginManager $dataTransferManager,
    protected LanguageManagerInterface $languageManager,
    protected FileSystemInterface $fileSystem,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected BabelStorageInterface $babelStorage,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $options = $extensions = [];
    foreach ($this->dataTransferManager->getImporters() as $pluginId => $definition) {
      $extensions[$pluginId] = $definition['fileExtensions'];
      $allowedExtensions[$pluginId] = implode(', ', array_map(
        fn(string $extension): string => ".$extension",
        $definition['fileExtensions'],
      ));
      $options[$pluginId] = $this->t('@label (@extensions)', [
        '@label' => $definition['label'],
        '@extensions' => $allowedExtensions[$pluginId],
      ]);
    }

    $languages = array_map(
      fn(LanguageInterface $language): string => $language->getName(),
      $this->languageManager->getLanguages(),
    );
    unset($languages['en']);

    $form['language'] = [
      '#type' => 'select',
      '#title' => $this->t('Language'),
      '#options' => $languages,
      '#required' => TRUE,
    ];
    $form['plugin'] = [
      '#type' => 'select',
      '#title' => $this->t('Importer'),
      '#options' => $options,
      '#required' => TRUE,
      '#ajax' => ['callback' => '::refreshFileUpload'],
    ];

    $pluginId = $form_state->getValue('plugin') ?? $form_state->getUserInput()['plugin'] ?? NULL;

    if ($pluginId) {
      $dataTransfer = $this->config('babel.settings')->get('data_transfer');

      $form['upload'] = [
        '#type' => 'managed_file',
        '#title' => $this->t('File to import'),
        '#upload_location' => $dataTransfer['destination'],
        '#upload_validators' => [
          'FileExtension' => ['extensions' => implode(' ', $extensions[$pluginId])],
        ],
        '#required' => TRUE,
        '#description' => $this->t('Allowed extensions: @extensions', [
          '@extensions' => $allowedExtensions[$pluginId],
        ]),
      ];
    }

    $form['actions'] = [
      '#type' => 'actions',
      'submit' => [
        '#type' => 'submit',
        '#value' => $this->t('Import'),
      ],
    ];

    if ($this->isAjax()) {
      // @see https://www.drupal.org/node/2897377
      $form['#id'] = Html::getId($form_state->getBuildInfo()['form_id']);
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $langcode = $form_state->getValue('language');
    $file = $this->entityTypeManager->getStorage('file')->load($form_state->getValue(['upload', 0]));

    if (!file_exists($file->getFileUri())) {
      // Deleted in the meantime?
      throw new \RuntimeException("File {$file->getFileUri()} not found");
    }

    $plugin = $this->dataTransferManager->createInstance($form_state->getValue('plugin'));
    $path = $this->fileSystem->realpath($file->getFileUri());
    $translations = $plugin->getImportedTranslations($path, $langcode);
    $file->delete();

    $arguments = [
      '%file' => $file->label(),
      '%lang' => $this->languageManager->getLanguage($langcode)->getName(),
      '@type' => $plugin->getPluginDefinition()['label'],
    ];

    if ($errors = $plugin->getImportErrors()) {
      static::message(
        MessengerInterface::TYPE_ERROR,
        $this->t('Importing data from from the %file @type file into the %lang translation failed with the following errors:', $arguments),
        $errors,
      );
      return;
    }

    $translations = array_map(
      function (array $translation): string {
        $translation = implode(PoItem::DELIMITER, array_map('trim', $translation));
        // Don't store unneeded empty variants.
        // If none is needed, then the translation string will be empty.
        return rtrim($translation, PoItem::DELIMITER);
      },
      $translations,
    );
    // Ignore empty translations (instead of deleting them).
    $translations = array_filter($translations);

    $batch = (new BatchBuilder())
      ->addOperation([static::class, 'initImportTranslation'], [$langcode, $plugin->getImportWarnings(), $arguments])
      ->setFinishCallback([static::class, 'finishImportTranslation'])
      ->setTitle($this->t('Importing'))
      ->setErrorMessage($this->t('Failed to import %lang translations from the %file @type file.', $arguments));

    foreach ($translations as $hash => $translation) {
      foreach ($this->babelStorage->getSourceStringInstances($hash) as $pluginId => $ids) {
        foreach ($ids as $id) {
          $batch->addOperation([static::class, 'importTranslation'], [$pluginId, $id, $translation]);
        }
      }
    }

    batch_set($batch->toArray());
  }

  /**
   * Provided an Ajax submit callback.
   *
   * @param array $form
   *   The form render array.
   * @param \Drupal\Core\Form\FormStateInterface $formState
   *   The form state instance.
   *
   * @return \Drupal\Core\Ajax\AjaxResponse
   *   The Ajax response.
   */
  public function refreshFileUpload(array $form, FormStateInterface $formState): AjaxResponse {
    $selector = '[data-drupal-selector="' . $form['#attributes']['data-drupal-selector'] . '"]';
    return (new AjaxResponse())->addCommand(new ReplaceCommand($selector, $form));
  }

  /**
   * Initializes the import batch process.
   *
   * @param string $langcode
   *   The language code.
   * @param array $warnings
   *   List of warning messages gathered during import.
   * @param array $arguments
   *   Arguments to be used in messages.
   * @param array $context
   *   The batch process context.
   */
  public static function initImportTranslation(string $langcode, array $warnings, array $arguments, array &$context): void {
    $context['results']['langcode'] = $langcode;
    $context['results']['warnings'] = $warnings;
    $context['results']['arguments'] = $arguments;
    $context['results']['success'] = 0;
    $context['results']['deleted'] = [];
  }

  /**
   * Saves one translated string to the backend.
   *
   * @param string $pluginId
   *   The translation type plugin ID.
   * @param string $id
   *   The translation ID in the realm of $pluginId plugin.
   * @param string $translation
   *   The translation.
   * @param array $context
   *   The batch process context.
   */
  public static function importTranslation(string $pluginId, string $id, string $translation, array &$context): void {
    $plugin = \Drupal::service(TranslationTypePluginManager::class)->createInstance($pluginId);
    if ($string = $plugin->getString($context['results']['langcode'], $id)) {
      $string->source->setStatus(TRUE);
      $plugin->updateTranslation($string, $id, $context['results']['langcode'], $translation);
      if ($translation) {
        $context['results']['success']++;
      }
      else {
        $context['results']['deleted'][] = "$pluginId:$id";
      }
    }
  }

  /**
   * Finishes the import batch process.
   *
   * @param bool $success
   *   Whether the process was successful.
   * @param array $results
   *   The batch process results.
   */
  public static function finishImportTranslation(bool $success, array $results): void {
    $arguments = $results['arguments'] + [
      '@imported-translations' => $results['success'],
      '@deleted-translations' => implode(', ', $results['deleted']),
    ];

    if ($success) {
      \Drupal::messenger()->addStatus(
        $results['deleted']
          ? t("@imported-translations items imported to the %lang translation from the %file @type file.\nDeleted the translation of: @deleted-translations.", $arguments)
          : t('@imported-translations items imported to the %lang translation from the %file @type file.', $arguments),
      );

      if ($results['warnings']) {
        static::message(
          MessengerInterface::TYPE_WARNING,
          new PluralTranslatableMarkup(
            count($results['warnings']),
            '@count warning reported during import:',
            '@count warnings reported during import:',
          ),
          $results['warnings'],
        );
      }

      return;
    }

    \Drupal::messenger()->addError('Import failed');
  }

  /**
   * Pushes a status message.
   *
   * @param string $level
   *   The message level.
   * @param \Drupal\Core\StringTranslation\TranslatableMarkup $title
   *   The message itself.
   * @param array $messages
   *   (optional) Subsequent list of messages such as errors or warnings.
   */
  protected static function message(string $level, TranslatableMarkup $title, array $messages = []): void {
    $message = [
      [
        '#markup' => $title,
      ],
      [
        '#theme' => 'item_list',
        '#items' => $messages,
      ],
    ];
    \Drupal::messenger()->addMessage($message, $level);
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'babel_export_form';
  }

}
