<?php

declare(strict_types=1);

namespace Drupal\babel\Plugin\Babel\TranslationType;

use Drupal\babel\BabelStorageInterface;
use Drupal\babel\Model\Source;
use Drupal\babel\Model\StringTranslation;
use Drupal\babel\Model\Translation;
use Drupal\babel\Plugin\Babel\TranslationType;
use Drupal\babel\Plugin\Babel\TranslationTypePluginBase;
use Drupal\Component\Utility\DeprecationHelper;
use Drupal\Core\Database\Statement\FetchAs;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\locale\StringStorageInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Handles translatable strings managed by Locale module.
 */
#[TranslationType(
  id: 'locale',
  label: new TranslatableMarkup('Locale'),
)]
class Locale extends TranslationTypePluginBase {

  public function __construct(
    array $configuration,
    string $pluginId,
    array $pluginDefinition,
    BabelStorageInterface $babelStorage,
    protected readonly StringStorageInterface $storage,
  ) {
    parent::__construct($configuration, $pluginId, $pluginDefinition, $babelStorage);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): self {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get(BabelStorageInterface::class),
      $container->get('locale.storage'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getStrings(string $langcode, array $ids = []): array {
    $query = $this->babelStorage->getBaseQuery($this->getPluginId(), $ids);

    $query->innerJoin('locales_source', 'ls', 'bsi.id = ls.lid');
    $query->fields('ls', ['source', 'context']);
    if ($ids) {
      $query->condition('ls.lid', $ids, 'IN');
    }
    $query->leftJoin('locales_target', 'lt', 'ls.lid = lt.lid AND lt.language = :langcode', [':langcode' => $langcode]);
    $query->fields('lt', ['translation']);

    // @todo Remove when Drupal 11.2 is the minimum supported core version.
    $mode = DeprecationHelper::backwardsCompatibleCall(
      currentVersion: \Drupal::VERSION,
      deprecatedVersion: '11.2',
      // This enum only exists in Drupal core 11.2+.
      // @phpstan-ignore-next-line
      currentCallable: fn() => FetchAs::Associative,
      deprecatedCallable: fn() => \PDO::FETCH_ASSOC,
    );

    return array_map(
      function (array $row) use ($langcode): StringTranslation {
        $row['plugin'] = $this->getPluginId();
        $row['sortKey'] = $row['sort_key'];
        unset($row['sort_key']);

        return new StringTranslation(
          source: new Source(
            string: $row['source'],
            context: $row['context'],
            status: (bool) $row['status'],
          ),
          translation: $row['translation'] ? new Translation(
            language: $langcode,
            string: $row['translation'],
          ) : NULL,
        );
      },
      $query->execute()->fetchAllAssoc('id', $mode),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getString(string $langcode, string $id): ?StringTranslation {
    $query = $this->babelStorage->getBaseQuery($this->getPluginId(), fields: ['id'], includeSortKey: FALSE)->condition('bsi.id', $id);
    $query->innerJoin('locales_source', 'ls', 'bsi.id = ls.lid');
    $query->fields('ls', ['source', 'context']);
    $query->leftJoin('locales_target', 'lt', 'ls.lid = lt.lid AND lt.language = :langcode', [':langcode' => $langcode]);
    $query->fields('lt', ['translation', 'language']);

    if (!$values = $query->execute()->fetch()) {
      return NULL;
    }

    return new StringTranslation(
      source: new Source(
        string: $values->source,
        context: $values->context,
        status: (bool) $values->status,
      ),
      translation: $values->translation ? new Translation(
        language: $langcode,
        string: $values->translation,
      ) : NULL,
    );
  }

  /**
   * {@inheritdoc}
   */
  public function updateTranslation(StringTranslation $string, string $id, string $langcode, string $translation): void {
    if ($translation) {
      $this->storage->createTranslation([
        'lid' => $id,
        'language' => $langcode,
        'translation' => $translation,
      ])->save();
    }
    else {
      $this->storage->deleteTranslations([
        'lid' => $id,
        'language' => $string->getTranslation()->language ?? $langcode,
      ]);
    }

    _locale_refresh_translations([$string->getTranslation()?->language ?? $langcode], [$id]);
    _locale_rebuild_js($string->getTranslation()?->language ?? $langcode);

    parent::updateTranslation($string, $id, $langcode, $translation);
  }

}
