<?php

declare(strict_types=1);

namespace Drupal\babel_content_entity;

use Drupal\babel\BabelStorageInterface;
use Drupal\babel\Model\Source;
use Drupal\babel\Plugin\Babel\TranslationTypePluginManager;
use Drupal\babel\StringsCollectorFactory;
use Drupal\babel_content_entity\Plugin\Babel\TranslationType\ContentEntity;
use Drupal\Component\Plugin\PluginBase;
use Drupal\Core\Batch\BatchBuilder;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\ContentEntityTypeInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Field\Plugin\Field\FieldType\StringItemBase;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\text\Plugin\Field\FieldType\TextItemBase;

/**
 * Installation helper.
 */
class BabelContentEntityService {

  /**
   * List of the entity fields which aren't translated.
   *
   * @const string[]
   */
  public const EXCLUDED_FIELDS = [
    'content_translation_created',
    'content_translation_outdated',
    'content_translation_source',
    'content_translation_status',
    'content_translation_uid',
    'default_langcode',
    'langcode',
    'revision_translation_affected',
  ];

  public function __construct(
    protected readonly EntityTypeManagerInterface $entityTypeManager,
    protected readonly EntityTypeBundleInfoInterface $bundleInfo,
    protected readonly BabelStorageInterface $babelStorage,
    protected readonly LanguageManagerInterface $languageManager,
    protected readonly TranslationTypePluginManager $translationTypeManager,
    protected readonly EntityFieldManagerInterface $fieldManager,
    protected readonly StringsCollectorFactory $collectorFactory,
  ) {}

  /**
   * Updates the source objects for the entity.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The content entity to process.
   */
  public function updateSourcesForEntity(EntityInterface $entity): void {
    if (!$this->isEntityAllowed($entity)) {
      return;
    }

    $pluginId = ContentEntity::id($entity->getEntityTypeId());
    assert($entity instanceof ContentEntityInterface);

    $defaultLangcode = $this->languageManager->getDefaultLanguage()->getId();
    $langcode = $entity->language()->getId();

    // Translation update. We're only invalidating the cache.
    if ($langcode !== $defaultLangcode) {
      $query = $this->babelStorage->getBaseQuery(
        pluginId: $pluginId,
        fields: ['hash']
      );
      $hashes = $query
        ->condition('id', $query->escapeLike("{$entity->bundle()}:{$entity->id()}:") . '%', 'LIKE')
        ->execute()
        ->fetchCol();
      $this->collectorFactory->invalidateHashes($hashes, $langcode);
      return;
    }

    // Cleanup first to handle delta changes.
    $this->babelStorage->delete($pluginId, "{$entity->bundle()}:{$entity->id()}:");
    $this->babelStorage->update($pluginId, $this->getSourcesForEntity($entity));
  }

  /**
   * Instantiates source objects from the properties of the given entity.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The content entity to process.
   *
   * @return list<\Drupal\babel\Model\Source>
   *   List of source objects.
   */
  public function getSourcesForEntity(ContentEntityInterface $entity): array {
    $sources = [];

    if (!$this->isEntityAllowed($entity)) {
      return $sources;
    }

    $definitions = $this->getBundleTranslatableStringFields($entity->getEntityTypeId(), $entity->bundle());
    foreach ($definitions as $fieldName => $definition) {
      if (!$entity->hasField($fieldName) || ($field = $entity->get($fieldName))->isEmpty()) {
        continue;
      }

      $property = $definition->getFieldStorageDefinition()->getMainPropertyName();
      foreach ($field as $delta => $item) {
        $value = (string) $item->{$property};
        if (!trim($value)) {
          continue;
        }

        $id = $this->buildSourceInstanceId($entity, $fieldName, $delta);
        $sources[$id] = new Source(string: $value, context: '');
      }
    }

    return $sources;
  }

  /**
   * Creates a batch process to process preexisting content entities.
   *
   * @param string $entityTypeId
   *   The entity type ID.
   * @param array $bundles
   *   (optional) Bundles to be processed. If omitted all bundles are processed.
   */
  public function batchAddSources(string $entityTypeId, array $bundles = []): void {
    $query = $this->entityTypeManager->getStorage($entityTypeId)
      ->getQuery()
      ->accessCheck(FALSE);

    $entityType = $this->entityTypeManager->getDefinition($entityTypeId);

    if ($entityType->hasKey('bundle')) {
      $allBundles = array_keys($this->bundleInfo->getBundleInfo($entityTypeId));
      if ($bundles = ($bundles ? array_intersect($bundles, $allBundles) : $allBundles)) {
        $query->condition($entityType->getKey('bundle'), $bundles, 'IN');
      }
    }
    if (!$ids = $query->execute()) {
      return;
    }
    $total = count($ids);

    $batch = new BatchBuilder();
    foreach (array_chunk($ids, 10) as $chunk) {
      $batch->addOperation([BatchHelper::class, 'processContentEntities'], [$entityTypeId, $chunk, $total]);
    }

    batch_set($batch->toArray());
  }

  /**
   * Returns the bundle translatable string fields.
   *
   * @param string $entityTypeId
   *   The entity type ID.
   * @param string $bundle
   *   The bundle.
   *
   * @return array<non-empty-string, \Drupal\Core\Field\FieldDefinitionInterface>
   *   Bundle translatable string fields definitions keyed by field name.
   */
  public function getBundleTranslatableStringFields(string $entityTypeId, string $bundle): array {
    $translatable = [];

    $entityType = $this->entityTypeManager->getDefinition($entityTypeId);
    if (!$entityType instanceof ContentEntityTypeInterface || !$entityType->isTranslatable()) {
      return $translatable;
    }

    $fields = array_diff_key(
      $this->fieldManager->getFieldDefinitions($entityTypeId, $bundle),
      array_flip(BabelContentEntityService::EXCLUDED_FIELDS),
    );

    if (!$fields) {
      return $translatable;
    }

    foreach ($fields as $fieldName => $field) {
      if (!$field->isTranslatable()) {
        continue;
      }

      $class = $field->getItemDefinition()->getClass();

      if (
        is_a($class, StringItemBase::class, TRUE) ||
        is_a($class, TextItemBase::class, TRUE)
      ) {
        // Only string fields are accepted.
        $translatable[$fieldName] = $field;
      }
    }

    return $translatable;
  }

  /**
   * Builds a source instance ID.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The content entity.
   * @param string $fieldName
   *   The field name.
   * @param int|string $delta
   *   The field item delta.
   *
   * @return string
   *   The source instance ID.
   */
  public function buildSourceInstanceId(ContentEntityInterface $entity, string $fieldName, int|string $delta): string {
    return "{$entity->bundle()}:{$entity->id()}:$fieldName:$delta";
  }

  /**
   * Destructs a source instance ID.
   *
   * @param string $pluginId
   *   The plugin ID.
   * @param string $id
   *   The source instance ID.
   *
   * @return array{\Drupal\Core\Entity\ContentEntityInterface, string, int}
   *   Triplet: entity, field name, delta.
   */
  public function destructSourceInstanceId(string $pluginId, string $id): array {
    [, $entityTypeId] = explode(PluginBase::DERIVATIVE_SEPARATOR, $pluginId, 2);
    [, $entityId, $fieldName, $delta] = explode(':', $id);
    $entity = $this->entityTypeManager->getStorage($entityTypeId)->load($entityId);
    return [$entity, $fieldName, (int) $delta];
  }

  /**
   * Check whether the passed entity is handled by Babel.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The content entity.
   *
   * @return bool
   *   Whether the passed entity is handled by Babel.
   */
  protected function isEntityAllowed(EntityInterface $entity): bool {
    if (!$entity instanceof ContentEntityInterface) {
      return FALSE;
    }

    $entityTypeId = $entity->getEntityTypeId();
    $pluginId = ContentEntity::id($entityTypeId);

    if (!$this->translationTypeManager->hasDefinition($pluginId)) {
      return FALSE;
    }

    $plugin = $this->translationTypeManager->createInstance($pluginId);
    $bundles = $plugin->getConfiguration()['bundle'] ?: array_keys($this->bundleInfo->getBundleInfo($entityTypeId));

    return in_array($entity->bundle(), $bundles, TRUE);
  }

}
