<?php

declare(strict_types=1);

namespace Drupal\babel_content_entity\Plugin\Babel\TranslationType;

use Drupal\babel\BabelStorageInterface;
use Drupal\babel\Model\Source;
use Drupal\babel\Model\StringTranslation;
use Drupal\babel\Model\Translation;
use Drupal\babel\Plugin\Babel\TranslationType;
use Drupal\babel\Plugin\Babel\TranslationTypePluginBase;
use Drupal\babel\StringsCollectorFactory;
use Drupal\babel_content_entity\BabelContentEntityService;
use Drupal\babel_content_entity\Plugin\Babel\BabelContentEntityDeriver;
use Drupal\Component\Plugin\PluginBase;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\PluginFormInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Handles content entities translatable strings.
 */
#[TranslationType(
  id: self::PLUGIN_ID,
  label: new TranslatableMarkup('Content Entity'),
  deriver: BabelContentEntityDeriver::class,
)]
class ContentEntity extends TranslationTypePluginBase implements PluginFormInterface {

  protected const PLUGIN_ID = 'content_entity';

  public function __construct(
    array $configuration,
    string $pluginId,
    array $pluginDefinition,
    BabelStorageInterface $babelStorage,
    StringsCollectorFactory $collectorFactory,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected BabelContentEntityService $service,
    protected EntityTypeBundleInfoInterface $bundleInfo,
    protected EntityFieldManagerInterface $fieldManager,
  ) {
    parent::__construct($configuration, $pluginId, $pluginDefinition, $babelStorage, $collectorFactory);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): self {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get(BabelStorageInterface::class),
      $container->get(StringsCollectorFactory::class),
      $container->get('entity_type.manager'),
      $container->get(BabelContentEntityService::class),
      $container->get('entity_type.bundle.info'),
      $container->get('entity_field.manager'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return ['bundle' => []] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function getStrings(string $langcode, array $ids = []): array {
    if (!$ids) {
      $ids = $this->babelStorage->getBaseQuery(
        pluginId: $this->getPluginId(),
        includeSortKey: FALSE,
      )->execute()->fetchCol();
    }

    $entityIds = [];
    foreach ($ids as $id) {
      [$entity, $fieldName, $delta] = $this->service->destructSourceInstanceId($this->getPluginId(), $id);
      if ($entity) {
        $entityIds[$entity->id()][] = [$fieldName, $delta];
      }
    }

    if (!$entityIds) {
      return [];
    }

    $storage = $this->entityTypeManager->getStorage($this->getDerivativeId());

    $strings = [];
    foreach ($storage->loadMultiple(array_keys($entityIds)) as $entityId => $entity) {
      $hasTranslation = $entity->hasTranslation($langcode);
      if ($hasTranslation) {
        $translated = $entity->getTranslation($langcode);
      }

      foreach ($entityIds[$entityId] as [$fieldName, $delta]) {
        $source = $entity?->get($fieldName)?->get($delta)->getString();

        if (!$source) {
          continue;
        }

        $translatedString = $hasTranslation
          ? $translated?->get($fieldName)?->get($delta)?->getString()
          : NULL;
        $translation = $translatedString ? new Translation(
          language: $langcode,
          string: $translatedString,
        ) : NULL;
        $id = $this->service->buildSourceInstanceId($entity, $fieldName, $delta);
        $strings[$id] = new StringTranslation(
          source: new Source(string: $source, context: '', status: TRUE),
          translation: $translation,
        );
      }
    }

    return $strings;
  }

  /**
   * {@inheritdoc}
   */
  public function updateTranslation(StringTranslation $string, string $id, string $langcode, string $translation): void {
    [$entity, $fieldName, $delta] = $this->service->destructSourceInstanceId($this->getPluginId(), $id);

    if ($entity) {
      if ($entity->hasTranslation($langcode)) {
        $translated = $entity->getTranslation($langcode);
      }
      else {
        $translated = $entity->addTranslation($langcode, $entity->toArray());
      }

      $translated->get($fieldName)->setValue([$delta => $translation]);
      $translated->save();
    }

    parent::updateTranslation($string, $id, $langcode, $translation);
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $entityType = $this->entityTypeManager->getDefinition($this->getDerivativeId());

    $bundles = $this->getBundlesWithTranslatableStringFields();

    if (count($bundles) > 1) {
      $form['bundle'] = [
        '#type' => 'select',
        '#title' => $entityType->getBundleLabel(),
        '#description' => $this->t('No selection means "all"'),
        '#options' => $bundles,
        '#default_value' => $this->getConfiguration()['bundle'],
        '#multiple' => TRUE,
      ];
    }
    else {
      if ($entityType->hasKey('bundle')) {
        $bundle = reset($bundles);
        $form['help'] = [
          '#markup' => $this->t('The %label entity type has only one @name (%bundle) which will be used', [
            '%label' => $entityType->getSingularLabel(),
            '%bundle' => $bundle,
            '@name' => strtolower($entityType->getBundleLabel()),
          ]),
        ];
      }
      $form['bundle'] = [
        '#type' => 'value',
        '#value' => [],
      ];
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state): void {}

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    $configuration = $form_state->getValues();
    // Cleanup the result of the form element values.
    $configuration['bundle'] = array_keys(array_filter($configuration['bundle']));
    $this->setConfiguration($configuration);
  }

  /**
   * Utility method to build a derivative plugin ID given the entity type.
   *
   * @param string|null $entityTypeId
   *   The entity type ID. If omitted, only the plugin base ID is returned.
   *
   * @return string
   *   A full or base plugin ID.
   */
  public static function id(?string $entityTypeId = NULL): string {
    return static::PLUGIN_ID . ($entityTypeId !== NULL ? PluginBase::DERIVATIVE_SEPARATOR . $entityTypeId : '');
  }

  /**
   * Returns the bundles with translatable string fields.
   *
   * @return array<non-empty-string, string|\Stringable>
   *   Bundle labels keyed by bundle.
   */
  protected function getBundlesWithTranslatableStringFields(): array {
    $entityTypeId = $this->getDerivativeId();

    $bundles = [];
    foreach ($this->bundleInfo->getBundleInfo($entityTypeId) as $bundle => $info) {
      if ($this->service->getBundleTranslatableStringFields($entityTypeId, $bundle)) {
        $bundles[$bundle] = $info['label'] ?? $bundle;
      }
    }

    return $bundles;
  }

}
