<?php

declare(strict_types=1);

namespace Drupal\babel_menu_link_content;

use Drupal\babel\BabelStorageInterface;

/**
 * Batch operations callbacks.
 */
class BabelMenuLinkContentBatchHelper {

  /**
   * Batch callback to process menu link content entities.
   *
   * @param array<string|int> $lids
   *   ID of menu link content entities.
   * @param int $total
   *   Total number of items to process. Only used to construct the interactive
   *   message.
   * @param array $context
   *   The batch context.
   */
  public static function processMenuLinks(array $lids, int $total, array &$context): void {
    $defaultLanguage = \Drupal::languageManager()->getDefaultLanguage();
    $context['results']['progress'] ??= 0;

    $storage = \Drupal::entityTypeManager()->getStorage('menu_link_content');
    $service = \Drupal::service(BabelMenuLinkContentService::class);

    $sources = [];
    foreach ($storage->loadMultiple($lids) as $link) {
      // Avoid duplicate sources created from translations of menu links.
      if ($link->language()->getId() === $defaultLanguage->getId()) {
        $sources = [...$sources, ...$service->getSourcesForLink($link)];
      }
    }
    \Drupal::service(BabelStorageInterface::class)->update('menu_link_content', $sources);

    $context['results']['progress'] += count($lids);
    $context['message'] = t('Processed @progress out of @total menu links', [
      '@progress' => $context['results']['progress'],
      '@total' => $total,
    ]);
  }

}
