<?php

declare(strict_types=1);

namespace Drupal\babel;

use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Cache\CacheCollector;
use Drupal\Core\Config\StorageCacheInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\KeyValueStore\KeyValueFactoryInterface;
use Drupal\Core\KeyValueStore\KeyValueStoreInterface;
use Drupal\Core\Lock\LockBackendInterface;
use Drupal\Core\TypedData\TraversableTypedDataInterface;
use Drupal\Core\TypedData\TypedDataInterface;
use Symfony\Component\DependencyInjection\Attribute\Autowire;

/**
 * Calculates the translatable properties of a config object.
 *
 * Getting the config translatable properties is an expensive process, see
 * ::doGetTranslatableStrings(). For this reason, we store the results in the
 * key-value persistent store, but also cache it to take advantage of the fast
 * caching backends such as Redis or Memcache.
 *
 * This service architecture is highly inspired by the 'state' service.
 *
 * @see \Drupal\Core\State\State
 */
class BabelConfigTranslatables extends CacheCollector {

  /**
   * Config translatables store.
   */
  protected KeyValueStoreInterface $store;

  public function __construct(
    #[Autowire(service: 'cache.babel')]
    CacheBackendInterface $cache,
    #[Autowire(service: 'lock')]
    LockBackendInterface $lock,
    #[Autowire(service: 'keyvalue')]
    KeyValueFactoryInterface $keyValueFactory,
    protected readonly StorageCacheInterface $configStorage,
    protected readonly TypedConfigManagerInterface $typedConfigManager,
  ) {
    parent::__construct('config_translatables', $cache, $lock);
    $this->store = $keyValueFactory->get('babel.config_translatables');
  }

  /**
   * {@inheritdoc}
   */
  protected function resolveCacheMiss($key): array {
    $paths = $this->store->get($key);

    if ($paths === NULL) {
      $paths = [];
      if ($data = $this->configStorage->read($key)) {
        $this->doGetTranslatableStrings($paths, $key, $data);
        // Store expensive processed data to the key value store.
        $this->store->set($key, $paths);
      }
    }

    $this->storage[$key] = $paths;
    $this->persist($key);

    return $paths;
  }

  /**
   * {@inheritdoc}
   */
  public function set($key, $value): void {
    $this->store->set($key, $value);

    // If another request had a cache miss before this request, and also hasn't
    // written to cache yet, then it may already have read this value from the
    // database and could write that value to the cache to the end of the
    // request. To avoid this race condition, write to the cache immediately
    // after calling parent::set(). This allows the race condition detection in
    // CacheCollector::set() to work.
    parent::set($key, $value);

    $this->persist($key);
    static::updateCache();
  }

  /**
   * {@inheritdoc}
   */
  public function delete($key): void {
    $this->store->delete($key);
    parent::delete($key);
  }

  /**
   * Processes translatable data within a nested data structure.
   *
   * @param array<array-key, string> $paths
   *   Passed by reference and populated during recursion. Associative array
   *   where the key is the path to the translatable string, as a string with
   *   dot as separator, and the value is the translation context.
   * @param string $name
   *   The configuration object name.
   * @param array $data
   *   The configuration data as an array.
   * @param \Drupal\Core\TypedData\TypedDataInterface|null $element
   *   Typed sata element. Used internally.
   * @param array $path
   *   Path to one translatable property. Used internally.
   */
  protected function doGetTranslatableStrings(array &$paths, string $name, mixed $data, ?TypedDataInterface $element = NULL, array $path = []): void {
    $element ??= $this->typedConfigManager->createFromNameAndData($name, $data);

    if ($element instanceof TraversableTypedDataInterface) {
      foreach ($element as $key => $childElement) {
        // This is a mapping/sequence, descent to a deeper level.
        $this->doGetTranslatableStrings($paths, $name, $data, $childElement, [...$path, $key]);
      }
    }
    else {
      $string = $element->getValue();
      // Process if the value is a non-empty string.
      if (is_string($string) && trim($string)) {
        $definition = $element->getDataDefinition();
        if (!empty($definition['translatable'])) {
          $context = $definition['translation context'] ?? '';
          $paths[implode('.', $path)] = $context;
        }
      }
    }
  }

  /**
   * Filters out the Webform configurations.
   *
   * Webforms are defined in configuration which contain huge translatable
   * properties, quit impossible to translate as a single string. We exclude
   * them from the radar of the 'config' plugin because they need a special
   * treatment as separate plugins.
   *
   * @param list<string> $names
   *   List of configuration names passed by reference.
   *
   * @todo Consider removing this method in https://drupal.org/i/3539510, by
   *   allowing the site builders to exclude the webforms by configuration.
   */
  public static function excludeWebformConfigurations(array &$names): void {
    $names = array_filter(
      $names,
      fn(string $name): bool => !preg_match('~^webform.(settings|webform.|webform_options.)~', $name),
    );
  }

}
