<?php

declare(strict_types=1);

namespace Drupal\babel;

use Drupal\Component\Gettext\PoItem;
use Drupal\locale\LocaleLookup;

/**
 * Replacement for Locale's lookup class.
 *
 * @see https://www.drupal.org/project/drupal/issues/3533589
 */
class BabelLocaleLookup extends LocaleLookup {

  /**
   * {@inheritdoc}
   */
  public function resolveCacheMiss($offset) {
    $translation = $this->stringStorage->findTranslation([
      'language' => $this->langcode,
      'source' => $offset,
      'context' => $this->context,
    ]);

    if ($translation) {
      $value = !empty($translation->translation) ? $translation->translation : TRUE;
    }
    else {
      // We don't have the source string, update the {locales_source} table to
      // indicate the string is not translated.
      // NOTE: Next line is changed compared to the original class. We store the
      // newly created translation object.
      $translation = $this->stringStorage->createString([
        'source' => $offset,
        'context' => $this->context,
        'version' => \Drupal::VERSION,
      ])->addLocation('path', $this->requestStack->getCurrentRequest()->getRequestUri())->save();
      $value = TRUE;
    }

    // If there is no translation available for the current language then use
    // language fallback to try other translations.
    if ($value === TRUE) {
      $fallbacks = $this->languageManager->getFallbackCandidates([
        'langcode' => $this->langcode,
        'operation' => 'locale_lookup',
        'data' => $offset,
      ]);
      if (!empty($fallbacks)) {
        foreach ($fallbacks as $langcode) {
          $translation = $this->stringStorage->findTranslation([
            'language' => $langcode,
            'source' => $offset,
            'context' => $this->context,
          ]);

          if ($translation && !empty($translation->translation)) {
            $value = $translation->translation;
            break;
          }
        }
      }
    }

    if (is_string($value) && str_contains($value, PoItem::DELIMITER)) {
      // Community translations imported from localize.drupal.org as well as
      // migrated translations may contain @count[number].
      $value = preg_replace('!@count\[\d+\]!', '@count', $value);
    }

    // NOTE: Line changed compared to the original. Store also the source ID.
    $this->storage[$offset] = [$translation->lid, $value];
    // Disabling the usage of string caching allows a module to watch for
    // the exact list of strings used on a page. From a performance
    // perspective that is a terrible idea, so we have no user
    // interface for this. Be careful when turning this option off!
    if ($this->configFactory->get('locale.settings')->get('cache_strings')) {
      $this->persist($offset);
    }
    // NOTE: Line changed compared to the original. Return also the source ID.
    return [$translation->lid, $value];
  }

}
