<?php

declare(strict_types=1);

namespace Drupal\babel;

use Drupal\babel\Model\Source;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Language\LanguageInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Lock\LockBackendInterface;
use Drupal\locale\LocaleTranslation;
use Drupal\locale\StringStorageInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Replacement for the 'string_translator.locale.lookup' service's class.
 *
 * @see https://www.drupal.org/project/drupal/issues/3533589
 */
class BabelLocaleTranslation extends LocaleTranslation {

  /**
   * List of source strings.
   *
   * @var list<\Drupal\babel\Model\Source>
   */
  protected array $sources = [];

  /**
   * List of locale source IDs which should be updated.
   *
   * @var list<int>
   */
  protected array $sourceIds = [];

  public function __construct(
    StringStorageInterface $storage,
    CacheBackendInterface $cache,
    LockBackendInterface $lock,
    ConfigFactoryInterface $configFactory,
    LanguageManagerInterface $languageManager,
    RequestStack $requestStack,
    protected BabelStorageInterface $babelStorage,
  ) {
    parent::__construct($storage, $cache, $lock, $configFactory, $languageManager, $requestStack);
  }

  /**
   * {@inheritdoc}
   */
  public function getStringTranslation($langcode, $string, $context): string|false {
    // If the language is not suitable for locale module, just return.
    if ($langcode == LanguageInterface::LANGCODE_SYSTEM || ($langcode == 'en' && !$this->canTranslateEnglish())) {
      return FALSE;
    }
    // Strings are cached by langcode, context and roles, using instances of the
    // LocaleLookup class to handle string lookup and caching.
    if (!isset($this->translations[$langcode][$context])) {
      // NOTE: Line changed compared to Drupal\locale\LocaleTranslation.
      $this->translations[$langcode][$context] = new BabelLocaleLookup($langcode, $context, $this->storage, $this->cache, $this->lock, $this->configFactory, $this->languageManager, $this->requestStack);
    }
    // NOTE: Next line is changed compared to the original class.
    [$id, $translation] = $this->translations[$langcode][$context]->get($string);

    // NOTE: Next line is added compared to Drupal\locale\LocaleTranslation.
    $this->sources[$id] = new Source(string: $string, context: $context);

    // If the translation is TRUE, no translation exists, but that string needs
    // to be stored in the persistent cache for performance reasons (so for
    // example, we don't have hundreds of queries to locale tables on each
    // request). That cache is persisted when the request ends, and the lookup
    // service is destroyed.
    return $translation === TRUE ? FALSE : $translation;
  }

  /**
   * Schedule a translatable string to be added / updated to our list.
   *
   * @param string|int|array $id
   *   A single ID or list of string IDs in the locale_source table.
   */
  public function scheduleStringCopy(string|int|array $id): void {
    $this->sourceIds = array_merge(
      $this->sourceIds,
      (array) $id
    );
  }

  /**
   * Prepares the collected locale string IDs to be updated in Babel storage.
   */
  protected function prepareStringIdsForUpdate(): void {
    if (!$this->sourceIds) {
      return;
    }
    $sourceStrings = $this->storage->getStrings(['lid' => array_values($this->sourceIds)]);
    foreach ($sourceStrings as $sourceString) {
      if (!$sourceString->isSource()) {
        continue;
      }
      $this->sources[$sourceString->lid] = new Source($sourceString->source, $sourceString->context);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function destruct(): void {
    $this->prepareStringIdsForUpdate();
    $this->babelStorage->update('locale', $this->sources);

    parent::destruct();

    // Strings were processed, make sure it starts on clean on a new call.
    $this->translations = [];
  }

}
