<?php

declare(strict_types=1);

namespace Drupal\babel;

use Drupal\Core\Database\Query\SelectInterface;
use Drupal\Core\DestructableInterface;

/**
 * Interface for babel storage service.
 */
interface BabelStorageInterface extends DestructableInterface {

  /**
   * Adds new records in the {babel_source_instance} table.
   *
   * @param string $pluginId
   *   The plugin ID.
   * @param \Drupal\babel\Model\Source[] $sources
   *   List of source strings.
   */
  public function update(string $pluginId, array $sources): void;

  /**
   * Deletes records from the {babel_source_instance} table.
   *
   * @param string $pluginId
   *   The plugin ID.
   * @param string|null $startsWith
   *   (optional) If passed, limits the deletion to IDs with that prefix. This
   *   parameter is mutually exclusive with $ids.
   * @param list<string>|null $ids
   *   (optional) If passed, limits the deletion this list of IDs. This
   *   parameter is mutually exclusive with $startsWith.
   *
   * @throws \InvalidArgumentException
   *   When both, $startsWith and $ids are passed.
   */
  public function delete(string $pluginId, ?string $startsWith = NULL, ?array $ids = NULL): void;

  /**
   * Schedules deletion.
   *
   * Schedules deletion of rows from {babel_source_instance} table
   * by plugin and ID.
   *
   * @param string $pluginId
   *   The plugin ID.
   * @param list<string|int> $ids
   *   IDs of rows to delete.
   */
  public function scheduleDeleteById(string $pluginId, array $ids): void;

  /**
   * Updates status for a specific hash.
   *
   * @param string $hash
   *   The hash to update.
   * @param bool $status
   *   The new status value.
   */
  public function updateStatusForHash(string $hash, bool $status): void;

  /**
   * Get status for a specific hash.
   *
   * @param string $hash
   *   The hash to get.
   *
   * @return bool
   *   The status value for the hash.
   */
  public function getStatusForHash(string $hash): bool;

  /**
   * Returns a list of source string instances.
   *
   * @param string $hash
   *   The source string unique hash.
   *
   * @return array<array-key, list<string>>
   *   Associative array keyed by the plugin ID. Each value is a list of IDs
   *   for that plugin.
   */
  public function getSourceStringInstances(string $hash): array;

  /**
   * Checks whether the passed hash exists.
   *
   * @param string $hash
   *   The source string unique hash.
   *
   * @return bool
   *   Whether the passed hash exists.
   */
  public function hashExists(string $hash): bool;

  /**
   * Returns the base query to be used by the translation type plugins.
   *
   * @param string $pluginId
   *   The plugin ID to filter by.
   * @param list<string> $limitToIds
   *   Limit the results to the records having one of the given IDs.
   * @param list<string> $fields
   *   List of columns to fetch from babel_source_instance. Defaults to "id".
   * @param bool $includeSortKey
   *   Whether to include the sort_key field. Defaults to TRUE.
   *
   * @return \Drupal\Core\Database\Query\SelectInterface
   *   Base DB query.
   */
  public function getBaseQuery(string $pluginId, array $limitToIds = [], array $fields = ['id'], bool $includeSortKey = TRUE): SelectInterface;

}
