<?php

declare(strict_types=1);

namespace Drupal\babel;

use Drupal\babel\Model\StringTranslation;
use Drupal\Core\Database\Connection;

/**
 * Repository service for strings.
 */
class BabelStringsRepository implements BabelStringsRepositoryInterface {

  public function __construct(
    protected readonly Connection $db,
    protected readonly StringsCollectorFactory $collectorFactory,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function getStrings(
    string $langcode,
    ?bool $translationStatus = NULL,
    ?string $search = NULL,
    ?bool $sourceStatus = NULL,
  ): array {
    $stringsCollector = $this->collectorFactory->get($langcode);

    $query = $this->db->select('babel_source')
      ->fields('babel_source', ['hash', 'status'])
      ->orderBy('sort_key');

    // A specific status is requested by the caller.
    if (is_bool($sourceStatus)) {
      $query->condition('status', $sourceStatus);
    }

    $strings = [];
    foreach ($query->execute()->fetchAllKeyed() as $hash => $status) {
      $value = $stringsCollector->get($hash);
      if ($value === NULL) {
        // Safeguard against malformed cache entries.
        continue;
      }

      $string = StringTranslation::fromArray($value);
      $string->source->setStatus((bool) $status);

      if (
        $this->getTranslationConditions($string, $translationStatus) &&
        $this->getSearchCondition($string, $search)
      ) {
        $strings[$hash] = $string;
      }
    }

    return $strings;
  }

  /**
   * Checks if the 'translated string' condition is satisfied.
   *
   * @param \Drupal\babel\Model\StringTranslation $string
   *   The string object (translated or not).
   * @param bool|null $translationStatus
   *   The translation status to check:
   *   - TRUE: Only translated strings.
   *   - FALSE: Only untranslated strings.
   *   - NULL: Both translated and untranslated strings.
   *
   * @return bool
   *   Whether the 'translated string' condition is satisfied.
   */
  protected function getTranslationConditions(StringTranslation $string, ?bool $translationStatus): bool {
    return match(TRUE) {
      $translationStatus === NULL => TRUE,
      $translationStatus => $string->isTranslated(),
      !$translationStatus => !$string->isTranslated(),
    };
  }

  /**
   * Checks if the search for string condition is satisfied.
   *
   * @param \Drupal\babel\Model\StringTranslation $string
   *   The string object (translated or not).
   * @param string|null $search
   *   The string contained in source or translation to filter on. If NULL is
   *   passed no filtering occurs.
   *
   * @return bool
   *   Whether the search for string condition is satisfied.
   */
  protected function getSearchCondition(StringTranslation $string, ?string $search): bool {
    if ($search === NULL || !($search = trim($search))) {
      // Condition satisfied: No valid search string has been provided.
      return TRUE;
    }

    $search = mb_strtolower($search);
    $sourceContains = str_contains(mb_strtolower($string->source->string), $search);
    $translationContains = $string->isTranslated() && str_contains(mb_strtolower($string->getTranslation()->string), $search);

    return $sourceContains || $translationContains;
  }

}
