<?php

declare(strict_types=1);

namespace Drupal\babel\Form;

use Drupal\babel\Ajax\PushParametersCommand;
use Drupal\babel\BabelStorageInterface;
use Drupal\babel\BabelStringsRepositoryInterface;
use Drupal\babel\Model\StringTranslation;
use Drupal\babel\Plugin\Babel\TranslationTypePluginManager;
use Drupal\babel\Utility;
use Drupal\Component\Gettext\PoItem;
use Drupal\Component\Render\MarkupInterface;
use Drupal\Component\Utility\Html;
use Drupal\Component\Utility\Xss;
use Drupal\Core\Ajax\AjaxHelperTrait;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\PrependCommand;
use Drupal\Core\Ajax\ReplaceCommand;
use Drupal\Core\DependencyInjection\AutowireTrait;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormBuilderInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Pager\PagerManagerInterface;
use Drupal\Core\Url;

/**
 * Translation form.
 */
class BabelTranslateForm extends FormBase {

  use AjaxHelperTrait;
  use AutowireTrait;

  /**
   * Number of string to reveal on a single page.
   *
   * @const int
   */
  protected const LIMIT = 50;

  public function __construct(
    protected TranslationTypePluginManager $manager,
    protected LanguageManagerInterface $languageManager,
    protected BabelStringsRepositoryInterface $stringsRepository,
    protected BabelStorageInterface $babelStorage,
    protected PagerManagerInterface $pagerManager,
    protected FormBuilderInterface $formBuilder,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function buildForm(
    array $form,
    FormStateInterface $form_state,
    ?LanguageInterface $language = NULL,
  ): array {
    $this->resetPaginationForFilterOperations($form_state);

    $languageOptions = $this->getLanguageOptions();
    $selectedLangcode = $language ? $language->getId() : ($form_state->getValue('langcode', $this->getRequest()->query->get('langcode')) ?: key($languageOptions));
    $translationStatusOptions = $this->getTranslationStatusOptions();
    $translationStatus = $form_state->getValue('translated', $this->getRequest()->query->get('translated')) ?: key($translationStatusOptions);
    $search = $form_state->getValue('search', $this->getRequest()->query->get('search')) ?: '';
    $includeInactive = $form_state->getValue('include_inactive', $this->getRequest()->query->get('include_inactive')) ?? FALSE;

    $translationStatusArg = match($translationStatus) {
      'untranslated' => FALSE,
      'translated' => TRUE,
      default => NULL,
    };
    $stringStatus = $includeInactive ? NULL : TRUE;
    $all = $selectedLangcode
      ? $this->stringsRepository->getStrings($selectedLangcode, $translationStatusArg, $search, $stringStatus)
      : [];
    $pager = $this->pagerManager->getPager() ?: $this->pagerManager->createPager(count($all), static::LIMIT);
    if ($pager->getTotalItems() !== count($all)) {
      // We have to reinitialize the pager.
      // @see https://drupal.org/i/3388718
      $pager = $this->pagerManager->createPager(count($all), static::LIMIT);
    }
    $strings = array_slice($all, $pager->getCurrentPage() * static::LIMIT, static::LIMIT);
    $form_state->set('strings', $strings);

    $form['#attached']['library'][] = 'babel/translate-form';
    $form['#attached']['library'][] = 'babel/textarea-height';
    $form['#attached']['library'][] = 'babel/push_parameters';
    $form['#attached']['library'][] = 'babel/popstate_handler';
    $form['#attached']['library'][] = 'babel/copy_source';

    $form['filters'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => [
          'babel-filters',
          'clearfix',
          'container-inline',
        ],
      ],
    ];

    $form['filters']['stats'] = [
      '#title' => $this->t('Translated'),
      '#title_display' => 'invisible',
      '#type' => 'item',
      '#markup' => $this->formatPlural(
        $pager->getTotalItems(),
        'One item matches the criteria',
        '@count items match the criteria',
      ),
    ];

    $form['filters']['langcode'] = [
      '#type' => $language ? 'value' : 'select',
      '#title' => $this->t('Language'),
      '#default_value' => $selectedLangcode,
      '#options' => $languageOptions,
    ];

    $form['filters']['translated'] = [
      '#title' => $this->t('Type'),
      '#type' => 'select',
      '#options' => $translationStatusOptions,
      '#default_value' => $translationStatus,
    ];

    $form['filters']['search'] = [
      '#title' => $this->t('Contains'),
      '#title_display' => 'invisible',
      '#type' => 'search',
      '#size' => 30,
      '#placeholder' => $this->t('Search'),
      '#wrapper_attributes' => [
        'class' => ['babel-search'],
        'data-disable-refocus' => TRUE,
      ],
      '#default_value' => $search,
    ];

    $form['filters']['include_inactive'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show inactive'),
      '#default_value' => $includeInactive,
    ];

    $form['filters']['apply'] = [
      '#type' => 'submit',
      '#op' => 'filter',
      '#name' => 'filter',
      '#value' => $this->t('Filter'),
      '#ajax' => [
        'callback' => [static::class, 'ajaxFilterChange'],
        'progress' => ['message' => NULL, 'type' => 'fullscreen'],
        'event' => 'click',
      ],
      '#executes_submit_callback' => FALSE,
    ];

    $form['pager_top'] = $pagerRenderable = [
      '#type' => 'pager_babel',
      '#parameters' => [
        'langcode' => $language ? NULL : $selectedLangcode,
        'translated' => $translationStatus,
        'search' => $search,
        'include_inactive' => $includeInactive,
      ],
      '#ajax_links' => TRUE,
      '#route_name' => 'babel.ui_pager',
      '#route_parameters' => array_filter([
        'js' => 'nojs',
        'language' => $language?->getId(),
      ]),
    ];

    $form['list'] = [
      '#type' => 'table',
      '#header' => [
        ['data' => $this->t('Source'), 'class' => ['babel-source-col']],
        [
          'data' => ['#markup' => '<span class="visually-hidden">' . $this->t('Copy') . '</span>'],
          'class' => ['babel-copy-col', 'js-show'],
        ],
        ['data' => $this->t('Translation'), 'class' => ['babel-translation-col']],
        $this->t('Save'),
        $this->t('Status'),
      ],
      '#empty' => $selectedLangcode
        ? $this->t('No strings are matching the criteria.')
        : $this->t('No translatable languages available. <a href=":add_language">Add a language</a> first.', [
          ':add_language' => Url::fromRoute('entity.configurable_language.collection')->toString(),
        ]),
      '#attributes' => [
        'class' => ['babel-list'],
      ],
    ];

    foreach ($strings as $hash => $string) {
      $form['list'][$hash] = $this->buildTranslationTableRow($hash, $string, $selectedLangcode, $form_state->getUserInput()['list'] ?? []);
    }

    // @see https://www.drupal.org/node/2897377
    $form['#attributes']['data-babel-form-id'] = Html::getId($form_state->getBuildInfo()['form_id']);

    $form['pager_bottom'] = $pagerRenderable;

    // Stabilize action if form is rendered via AJAX pager links.
    $form['#action'] = Url::fromRoute('babel.ui', array_filter([
      'language' => $language?->getId(),
    ]))->toString();

    return $form;
  }

  /**
   * Title callback of the form.
   *
   * @param \Drupal\Core\Language\LanguageInterface|null $language
   *   The current configurable language, if there is any specified by the
   *   current path, or NULL if we show a translation language selector.
   *
   * @return \Drupal\Component\Render\MarkupInterface
   *   The title of the form.
   */
  public function title(?LanguageInterface $language = NULL): MarkupInterface {
    return $language
      ? $this->t('Translate @language', ['@language' => $language->getName()])
      : $this->t('Translate');
  }

  /**
   * List of the languages which accept translations.
   *
   * @return array<string, string>
   *   The human label of the languages, keyed by their machine name.
   */
  protected function getLanguageOptions(): array {
    $availableLanguages = $this->languageManager->getLanguages();
    if (
      array_key_exists('en', $availableLanguages) &&
      !$this->configFactory()->get('locale.settings')->get('translate_english')
    ) {
      unset($availableLanguages['en']);
    }
    return array_map(
      fn (LanguageInterface $language): string => $language->getName(),
      $availableLanguages,
    );
  }

  /**
   * The available translation status options.
   *
   * @return array<string, \Drupal\Component\Render\MarkupInterface>
   *   The human label of the translation options, keyed by their unique form
   *   value.
   *
   * @see \Drupal\locale\Form\TranslateFormBase::translateFilters()
   */
  protected function getTranslationStatusOptions(): array {
    return [
      'all' => $this->t('Both translated and untranslated strings'),
      'untranslated' => $this->t('Only untranslated strings'),
      'translated' => $this->t('Only translated strings'),
    ];
  }

  /**
   * Builds a table form row from the given keys and StringTranslation object.
   *
   * @param string $hash
   *   A hash identifying a unique source string.
   * @param \Drupal\babel\Model\StringTranslation $string
   *   The StringTranslation object representation of the source string and the
   *   preexisting translation (if any).
   * @param string $selectedLangcode
   *   The language code of the selected translation language.
   * @param array $list
   *   The translations coming from the submitted form. This
   *   value comes from $form_state->getUserInput()['list'] ?? [].
   *
   * @return array[]
   *   Markup of the form table row.
   */
  protected function buildTranslationTableRow(string $hash, StringTranslation $string, string $selectedLangcode, array $list): array {
    $row = [
      '#attributes' => [
        'data-babel-row' => $hash,
        'class' => ['babel-translate-row'],
      ],
      'source' => ['#theme_wrappers' => ['container']],
    ];

    $sourceVariants = $string->getSourcePluralVariants();
    $sourceIsPlural = count($sourceVariants) > 1;

    foreach ($sourceVariants as $sourceVariantIndex => $variant) {
      $row['source'][$sourceVariantIndex] = [
        '#type' => 'textarea',
        '#title' => ($sourceVariantIndex ? $this->t('Plural form') : $this->t('Singular form')),
        '#title_display' => $sourceIsPlural ? 'before' : 'invisible',
        '#disabled' => TRUE,
        '#default_value' => $variant,
        '#wrapper_attributes' => ['data-babel-source-variant' => $sourceVariantIndex],
        '#attributes' => [
          'data-babel-textarea-height' => TRUE,
          'data-babel-textarea-height-group' => "$hash-$sourceVariantIndex",
        ],
      ];
    }
    if ($string->source->context) {
      $row['source'][] = [
        '#type' => 'html_tag',
        '#tag' => 'small',
        '#value' => $string->source->context,
        '#attributes' => ['class' => ['babel-context']],
      ];
    }

    $row['copy'] = [
      '#type' => 'button',
      '#value' => '',
      '#name' => 'babel-copy-' . $hash,
      '#attributes' => [
        'class' => ['babel-copy-button'],
        'title' => $this->t('Copy source to translation'),
        'data-babel-copy' => $hash,
        'type' => 'button',
      ],
      '#wrapper_attributes' => ['class' => ['babel--copy', 'js-show']],
    ];

    $row['translation'] = [
      '#type' => 'container',
      '#wrapper_attributes' => [
        'data-babel-translation' => $hash,
      ],
    ];
    // We need this, to detect changes in translations in the form.
    $row['translation']['translation_original'] = [];

    $translationVariants = $string->getTranslatedPluralVariants() ?: [];
    // StringTranslationTrait::getNumberOfPlurals can return string, so casting
    // here is truly necessary.
    $targetPluralVariants = (int) $this->getNumberOfPlurals($selectedLangcode);
    $variantNum = $sourceIsPlural ? $targetPluralVariants : 1;
    for ($targetVariantIndex = 0; $targetVariantIndex < $variantNum; $targetVariantIndex++) {

      // If the there was a change in the form, we use that translation.
      if (@$list[$hash]['translation']['translation_original'][$selectedLangcode][$targetVariantIndex] != @$list[$hash]['translation'][$selectedLangcode][$targetVariantIndex]) {
        $value = $list[$hash]['translation'][$selectedLangcode][$targetVariantIndex];
      }
      // Otherwise, we use the value coming from the storage, which
      // may be different from the one initially existing in the form.
      else {
        $value = $translationVariants[$targetVariantIndex] ?? '';
      }

      $row['translation'][$selectedLangcode][$targetVariantIndex] = [
        '#type' => 'textarea',
        '#title' => $this->getTranslationLabel($targetVariantIndex, $targetPluralVariants),
        '#title_display' => $sourceIsPlural ? 'before' : 'invisible',
        '#value' => $value,
        '#wrapper_attributes' => ['data-babel-translation-variant' => $targetVariantIndex],
        '#rows' => min(count(explode("\n", $variants[$targetVariantIndex] ?? $translationVariants[$targetVariantIndex] ?? '')), 8) ?: 1,
        '#attributes' => [
          'data-babel-textarea-height' => TRUE,
          'data-babel-textarea-height-group' => "$hash-$targetVariantIndex",
        ],
      ];
      $row['translation']['translation_original'][$selectedLangcode][$targetVariantIndex] = [
        '#type' => 'hidden',
        '#value' => $translationVariants[$targetVariantIndex] ?? '',
      ];
    }

    $row['save'] = [
      '#type' => 'submit',
      '#name' => 'babel-save-' . $hash,
      '#op' => 'update_translation',
      '#attributes' => [
        'data-babel-save' => TRUE,
      ],
      '#wrapper_attributes' => ['class' => ['babel--save']],
      '#has_garbage_value' => TRUE,
      '#value' => '',
      '#ajax' => [
        'callback' => '::ajaxUpdateTranslation',
        'progress' => ['type' => 'fullscreen', 'message' => NULL],
      ],
    ];

    $status = $string->source->getStatus();
    $activation_class = $status ? ['use-ajax-submit', 'activated'] : ['use-ajax-submit'];
    $activation_title = $status ? $this->t('Deactivate') : $this->t('Activate');

    $row['status'] = [
      '#type' => 'submit',
      '#value' => '',
      '#name' => 'babel-activate-' . $string->source->getHash(),
      '#op' => ['::status_change'],
      '#attributes' => [
        'class' => $activation_class,
        'title' => $activation_title,
      ],
      '#wrapper_attributes' => [
        'class' => ['babel--activate'],
      ],
      '#ajax' => [
        'callback' => '::ajaxToggleStatus',
        'wrapper' => 'babel-list',
        'progress' => ['type' => 'fullscreen', 'message' => NULL],
      ],
    ];

    return $row;
  }

  /**
   * Creates the translation element label.
   *
   * @param int $count
   *   The count that determines the label.
   * @param int $languagePluralVariants
   *   The number of plural variants for the current language.
   *
   * @return \Drupal\Component\Render\MarkupInterface
   *   The element label.
   */
  protected function getTranslationLabel(int $count, int $languagePluralVariants): MarkupInterface {
    return match(TRUE) {
      $count === 0 => $this->t('Singular form'),
      $count > 0 && $languagePluralVariants === 2 => $this->t('Plural form'),
      $count > 0 && $languagePluralVariants > 2 => $this->formatPlural($count, 'First plural form', '@ordinal plural form', [
        '@ordinal' => Utility::getOrdinal($count),
      ]),
    };
  }

  /**
   * Saves an updated translation.
   *
   * @param array $form
   *   Form render array.
   * @param \Drupal\Core\Form\FormStateInterface $formState
   *   Current form state.
   *
   * @return \Drupal\Core\Ajax\AjaxResponse
   *   Array of ajax commands to execute on submit of the modal form.
   */
  public function ajaxUpdateTranslation(array $form, FormStateInterface $formState): AjaxResponse {
    $triggeringElement = $formState->getTriggeringElement();
    $hash = array_reverse($triggeringElement['#array_parents'])[1] ?? NULL;
    $langcode = $formState->getValue('langcode');
    $formState->setRebuild();
    $response = new AjaxResponse();

    if ($hash && !empty($formState->getValue(['list', $hash]))) {
      $this->updateTranslation(
        $hash,
        $formState->getValue(['list', $hash, 'translation', $langcode]),
        $formState->get('strings')[$hash],
        $langcode,
      );
    }

    $formSelector = 'form[data-babel-form-id="' . $form['#attributes']['data-babel-form-id'] . '"]';
    $form = $this->formBuilder->rebuildForm($this->getFormId(), $formState, $form);

    return $response
      ->addCommand(new ReplaceCommand($formSelector, $form))
      ->addCommand(new PrependCommand($formSelector, ['#type' => 'status_messages']));
  }

  /**
   * Ajax callback to perform source string status change operation.
   *
   * @param array $form
   *   The form structure.
   * @param \Drupal\Core\Form\FormStateInterface $formState
   *   The form state.
   *
   * @return \Drupal\Core\Ajax\AjaxResponse
   *   The AJAX response.
   */
  public function ajaxToggleStatus(array $form, FormStateInterface $formState): AjaxResponse {
    $triggeringElement = $formState->getTriggeringElement();
    $hash = array_reverse($triggeringElement['#array_parents'])[1] ?? NULL;
    $string = $formState->get('strings')[$hash] ?? NULL;
    $formState->setRebuild();
    $response = new AjaxResponse();

    if ($string) {
      $this->toggleSourceStringStatus($string);
    }

    $formSelector = 'form[data-babel-form-id="' . $form['#attributes']['data-babel-form-id'] . '"]';
    $form = $this->formBuilder->rebuildForm($this->getFormId(), $formState, $form);

    return $response
      ->addCommand(new ReplaceCommand($formSelector, $form))
      ->addCommand(new PrependCommand($formSelector, ['#type' => 'status_messages']));
  }

  /**
   * Ajax callback of filtering actions.
   *
   * @param array $form
   *   The form structure.
   * @param \Drupal\Core\Form\FormStateInterface $formState
   *   The form state.
   *
   * @return \Drupal\Core\Ajax\AjaxResponse
   *   The AJAX response with the commands to apply on the current page.
   */
  public static function ajaxFilterChange(array $form, FormStateInterface $formState): AjaxResponse {
    $response = new AjaxResponse();
    $formState->setRebuild();

    $response->addCommand(new PushParametersCommand([
      'langcode' => $formState->getValue('langcode'),
      'translated' => $formState->getValue('translated'),
      'search' => $formState->getValue('search'),
      'include_inactive' => $formState->getValue('include_inactive'),
    ]));
    $response->addCommand(new ReplaceCommand('form[data-babel-form-id="' . $form['#attributes']['data-babel-form-id'] . '"]', $form));
    return $response;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    if ($this->isAjax()) {
      return;
    }

    $trigger = $form_state->getTriggeringElement();
    $operation = $trigger['#op']
      ? explode(':', $trigger['#op'], 2)[0]
      : NULL;
    $langcode = $form_state->getValue('langcode');

    switch ($operation) {
      case 'filter':
        $form_state->setRedirect(
          route_name: 'babel.ui',
          options: [
            'query' => array_filter([
              'langcode' => $langcode,
              'translated' => $form_state->getValue('translated'),
              'search' => $form_state->getValue('search'),
              'include_inactive' => $form_state->getValue('include_inactive'),
            ]),
          ],
        );
        return;

      case 'update_translation':
        // Keep the potentially changed translation values in other table rows.
        $form_state->setRebuild();

        $stringKey = array_reverse($trigger['#array_parents'])[1];
        $string = $form_state->get('strings')[$stringKey];
        $this->updateTranslation(
          $stringKey,
          $form_state->getValue(['list', $stringKey, 'translation', $langcode]),
          $string,
          $langcode,
        );
        return;

      case 'status_change':
        $form_state->setRebuild();

        $stringKey = array_reverse($trigger['#array_parents'])[1];
        $string = $form_state->get('strings')[$stringKey];
        $this->toggleSourceStringStatus($string);
        return;
    }
  }

  /**
   * Updates the translation belonging to the string with the given keys.
   *
   * @param string $hash
   *   A hash identifying a unique source string.
   * @param string[] $newTranslation
   *   The new translation of the source string.
   * @param \Drupal\babel\Model\StringTranslation $string
   *   The StringTranslation object representation of the source string and the
   *   preexisting translation (if any).
   * @param string $languageCode
   *   The language code of the translation.
   */
  protected function updateTranslation(
    string $hash,
    array $newTranslation,
    StringTranslation $string,
    string $languageCode,
  ): void {
    $sources = explode(PoItem::DELIMITER, $string->source->string);
    $sourceString = Xss::filter(implode(', ', $sources));
    $sourceStringShortened = mb_strlen($sourceString) > 80
      ? mb_substr($sourceString, 0, 79) . '…'
      : $sourceString;
    $messageArguments = [
      '%source' => $sourceStringShortened,
      '@language' => $this->languageManager->getLanguage($languageCode)->getName(),
    ];

    $translation = implode(PoItem::DELIMITER, array_map('trim', $newTranslation));
    // Don't store unneeded empty variants.
    // If none is needed, then the translation string will be empty.
    $translation = rtrim($translation, PoItem::DELIMITER);
    if ($translation === $string->getTranslation()?->string) {
      // No change was detected.
      $this->messenger()->addStatus(
        $this->formatPlural(
          count($sources),
          'No change was detected at string "%source".',
          'No change was detected at strings "%source".',
          $messageArguments,
        )
      );
      return;
    }

    foreach ($this->babelStorage->getSourceStringInstances($hash) as $pluginId => $ids) {
      $plugin = $this->manager->createInstance($pluginId);
      foreach ($ids as $id) {
        // Save (or remove) the translation.
        $plugin->updateTranslation($string, $id, $languageCode, $translation);
      }
    }

    // Build status feedback for the user.
    $statusMessage = $string->getTranslation()?->string
      ? $this->formatPlural(
        count($sources),
        'Translation of source string "%source" in language @language was updated.',
        'Translations of source strings "%source" in language @language were updated.',
        $messageArguments,
      )
      : $this->formatPlural(
        count($sources),
        'Translation of source string "%source" in language @language was deleted.',
        'Translations of source strings "%source" in language @language were deleted.',
        $messageArguments,
      );

    $this->messenger()->addStatus($statusMessage);
  }

  /**
   * Saves an updated translation activation state.
   *
   * @param \Drupal\babel\Model\StringTranslation $string
   *   The string translation object instance which source status must be
   *   changed.
   */
  protected function toggleSourceStringStatus(StringTranslation $string): void {
    $string->source->toggleStatus();
    $this->babelStorage->updateStatusForHash($string->source->getHash(), $string->source->getStatus());
  }

  /**
   * Resets pagination when filter operations are detected.
   */
  protected function resetPaginationForFilterOperations(FormStateInterface $form_state): void {
    $input = $form_state->getUserInput();
    $isFilterOperation = isset($input['_triggering_element_name']) &&
      $input['_triggering_element_name'] === 'filter';

    if ($isFilterOperation) {
      $this->getRequest()->query->set('page', 0);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'babel_translate';
  }

}
