<?php

declare(strict_types=1);

namespace Drupal\babel\Plugin\Babel\TranslationType;

use Drupal\babel\BabelConfigTranslatables;
use Drupal\babel\BabelStorageInterface;
use Drupal\babel\Model\Source;
use Drupal\babel\Model\StringTranslation;
use Drupal\babel\Model\Translation;
use Drupal\babel\Plugin\Babel\TranslationType;
use Drupal\babel\Plugin\Babel\TranslationTypePluginBase;
use Drupal\babel\StringsCollectorFactory;
use Drupal\Component\Utility\NestedArray;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\StorageCacheInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\language\ConfigurableLanguageManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Handles translatable strings from configuration objects.
 */
#[TranslationType(
  id: 'config',
  label: new TranslatableMarkup('Configuration'),
)]
class Config extends TranslationTypePluginBase {

  public function __construct(
    array $configuration,
    string $pluginId,
    array $pluginDefinition,
    BabelStorageInterface $babelStorage,
    StringsCollectorFactory $collectorFactory,
    protected readonly StorageCacheInterface $storage,
    protected readonly ConfigurableLanguageManagerInterface $languageManager,
    protected readonly ConfigFactoryInterface $configFactory,
    protected readonly BabelConfigTranslatables $configTranslatables,
  ) {
    parent::__construct($configuration, $pluginId, $pluginDefinition, $babelStorage, $collectorFactory);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): self {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get(BabelStorageInterface::class),
      $container->get(StringsCollectorFactory::class),
      $container->get('config.storage'),
      $container->get('language_manager'),
      $container->get('config.factory'),
      $container->get(BabelConfigTranslatables::class),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getStrings(string $langcode, array $ids = []): array {
    // @todo Consider replacing and simplifying this with high level API calls
    //   in https://drupal.org/i/3535124.
    if (!$ids) {
      $ids = $this->babelStorage->getBaseQuery($this->getPluginId())->execute()->fetchAllKeyed(0, 0);
    }

    $list = $pathsByName = [];
    foreach ($ids as $id) {
      [$name, $path] = explode(':', $id, 2);
      $pathsByName[$name][] = explode('.', $path);
    }

    $names = array_keys($pathsByName);
    $sources = $this->storage->readMultiple($names);
    $storage = $this->storage->createCollection("language.$langcode");
    $translations = $storage->readMultiple($names) ?: [];

    foreach ($pathsByName as $name => $paths) {
      // Skip if configuration doesn't exist.
      if (!isset($sources[$name])) {
        continue;
      }

      $translatables = $this->configTranslatables->get($name);
      foreach ($paths as $path) {
        $configTranslations = $translations[$name] ?? [];
        $translation = NestedArray::getValue($configTranslations, $path);
        $pathAsString = implode('.', $path);

        $list["$name:$pathAsString"] = new StringTranslation(
          source: new Source(
            string: NestedArray::getValue($sources[$name], $path),
            context: $translatables[$pathAsString],
          ),
          translation: $translation ? new Translation(
            language: $langcode,
            string: $translation,
          ) : NULL,
        );
      }
    }

    return $list;
  }

  /**
   * {@inheritdoc}
   */
  public function updateTranslation(StringTranslation $string, string $id, string $langcode, string $translation): void {
    [$name, $path] = explode(':', $id, 2);
    $override = $this->languageManager->getLanguageConfigOverride($langcode, $name);
    if ($translation) {
      $override->set($path, $translation);
    }
    else {
      $override->clear($path);
    }
    $override->save();
    $this->configFactory->reset();

    parent::updateTranslation($string, $id, $langcode, $translation);
  }

}
