<?php

declare(strict_types=1);

namespace Drupal\Tests\babel\FunctionalJavascript;

use Drupal\babel\BabelStorageInterface;
use Drupal\babel\Model\Source;
use Drupal\Component\Gettext\PoItem;
use Drupal\Core\Url;

/**
 * Tests the copy source to translation button functionality.
 *
 * @group babel
 */
class CopySourceToTranslationTest extends BabelFunctionalJavascriptTestBase {

  /**
   * Tests copying plural strings in LTR language.
   */
  public function testCopySourceToTranslation(): void {
    // Create a plural string in the locale storage.
    $localeStorage = \Drupal::service('locale.storage');
    $source_string = '1 person' . PoItem::DELIMITER . '@count persons';
    $localeSource = $localeStorage->createString(['source' => $source_string, 'context' => '']);
    $localeSource->save();

    // Register the string with Babel.
    $babelStorage = \Drupal::service(BabelStorageInterface::class);
    $source = new Source($source_string, '');
    $babelStorage->update('locale', [(string) $localeSource->lid => $source]);

    $this->drupalLogin($this->testAccount);
    $this->drupalGet(Url::fromRoute('babel.ui'));

    $page = $this->getSession()->getPage();
    $assertSession = $this->assertSession();

    $page->fillField('Search', 'person');
    $page->pressButton('Filter');
    $assertSession->assertWaitOnAjaxRequest();

    $singularSource = $page->find('xpath', '//*[@data-babel-source-variant="0"][contains(.,"1 person")]');
    $this->assertNotNull($singularSource);

    $translationRow = $singularSource->find('xpath', 'ancestor::tr');
    $pluralSource = $translationRow->find('xpath', '//*[@data-babel-source-variant="1"][contains(.,"@count persons")]');
    $this->assertNotNull($pluralSource);

    $singularTranslation = $translationRow->find('xpath', '//*[@data-babel-translation-variant="0"]//textarea');
    $pluralTranslation = $translationRow->find('xpath', '//*[@data-babel-translation-variant="1"]//textarea');
    $copyButton = $translationRow->find('css', '.babel-copy-button');

    $this->assertNotNull($copyButton);
    // Check that the copy button shows the right arrow icon.
    $copyButtonBackgroundImage = $this->getSession()->evaluateScript(
      'window.getComputedStyle(document.querySelector(".babel-copy-button")).getPropertyValue("background-image")'
    );
    $this->assertStringContainsString('arrow_right_bebebe.svg', $copyButtonBackgroundImage);
    $this->assertEmpty($singularTranslation->getValue());
    $this->assertEmpty($pluralTranslation->getValue());

    $copyButton->click();
    $this->getSession()->wait(200);

    $this->assertEquals('1 person', $singularTranslation->getValue());
    $this->assertEquals('@count persons', $pluralTranslation->getValue());

    // Test that clicking copy button again doesn't overwrite existing content.
    $singularTranslation->setValue('Custom translation');
    $pluralTranslation->setValue('Custom plural translation');

    $copyButton->click();
    $this->getSession()->wait(200);

    // Values should remain unchanged.
    $this->assertEquals('Custom translation', $singularTranslation->getValue());
    $this->assertEquals('Custom plural translation', $pluralTranslation->getValue());
  }

  /**
   * Tests copying plural strings in RTL language.
   */
  public function testCopySourceToTranslationRtl(): void {
    // Create an RTL language (Arabic).
    \Drupal::service('module_installer')->install(['language']);
    $language = \Drupal::service('entity_type.manager')
      ->getStorage('configurable_language')
      ->create([
        'id' => 'ar',
        'label' => 'Arabic',
        'direction' => 'rtl',
      ]);
    $language->save();

    // Create a plural string in the locale storage.
    $localeStorage = \Drupal::service('locale.storage');
    $source_string = '1 person' . PoItem::DELIMITER . '@count persons';
    $localeSource = $localeStorage->createString(['source' => $source_string, 'context' => '']);
    $localeSource->save();

    // Register the string with Babel.
    $babelStorage = \Drupal::service(BabelStorageInterface::class);
    $source = new Source($source_string, '');
    $babelStorage->update('locale', [(string) $localeSource->lid => $source]);

    $this->drupalLogin($this->testAccount);
    $this->drupalGet(Url::fromRoute('babel.ui', ['language' => 'ar']));

    $page = $this->getSession()->getPage();
    $assertSession = $this->assertSession();

    // Manually set the dir attribute to rtl for testing the CSS behavior.
    $this->getSession()->executeScript('document.documentElement.setAttribute("dir", "rtl");');

    $page->fillField('Search', 'person');
    $page->pressButton('Filter');
    $assertSession->assertWaitOnAjaxRequest();

    $singularSource = $page->find('xpath', '//*[@data-babel-source-variant="0"][contains(.,"1 person")]');
    $this->assertNotNull($singularSource);

    $translationRow = $singularSource->find('xpath', 'ancestor::tr');
    $copyButton = $translationRow->find('css', '.babel-copy-button');

    $this->assertNotNull($copyButton);
    // Check that the dir attribute is set to rtl on the html element.
    $dir = $this->getSession()->evaluateScript('document.documentElement.getAttribute("dir")');
    $this->assertEquals('rtl', $dir);

    // Check that the copy button shows the left arrow icon for RTL.
    $copyButtonBackgroundImage = $this->getSession()->evaluateScript(
      'window.getComputedStyle(document.querySelector(".babel-copy-button")).getPropertyValue("background-image")'
    );
    $this->assertStringContainsString('arrow_left_bebebe.svg', $copyButtonBackgroundImage);
  }

}
