<?php

declare(strict_types=1);

namespace Drupal\Tests\babel\Traits;

use Drupal\Component\Plugin\ConfigurableInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\language\ConfigurableLanguageInterface;
use Drupal\language\Entity\ConfigurableLanguage;

/**
 * Trait for setting up functional Babel tests.
 */
trait BabelFunctionalSetupTrait {

  use BatchTrait;

  /**
   * The language used for testing.
   *
   * @var \Drupal\language\ConfigurableLanguageInterface
   */
  protected ConfigurableLanguageInterface $language;

  /**
   * The user account used for testing.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected AccountInterface $testAccount;

  /**
   * Blocks to place into the "content" region if the current theme.
   *
   * If the value is an array, then it is handled as a block plugin ID. In this
   * case, the key must be integer, used as the weight of the block
   * configuration entity. If the value is an array it is used as a block
   * configuration; in this case, having a "plugin" key is mandatory.
   *
   * @var array<string|array>
   */
  protected array $blocksToPlace = [
    'page_title_block',
    'system_messages_block',
    'system_main_block',
    'system_powered_by_block',
  ];

  /**
   * Prepares the Drupal instance to Babel tests.
   */
  protected function babelTestSetup(): void {
    // Run batches set during module install.
    $this->runPendingBatches();

    $this->language = $this->createTestLanguage();
    $this->placeDefaultBlocks();

    $this->testAccount = $this->drupalCreateUser(static::TEST_USER_PERMISSIONS);
  }

  /**
   * Creates a configurable language and rebuild Drupal cache.
   *
   * @param string|null $langcode
   *   The language code of the language.
   *
   * @return \Drupal\language\ConfigurableLanguageInterface
   *   The saved language instance.
   */
  protected function createTestLanguage(?string $langcode = NULL): ConfigurableLanguageInterface {
    $langcode ??= static::LANGCODE;
    $language = ConfigurableLanguage::createFromLangcode($langcode);
    $language->save();
    // Rebuild container.
    \Drupal::getContainer()->get('kernel')->rebuildContainer();

    return $language;
  }

  /**
   * Places the blocks as declared in the blocksToPlace property.
   */
  protected function placeDefaultBlocks(): void {
    $pluginManager = \Drupal::service('plugin.manager.block');
    foreach ($this->blocksToPlace as $weight => $plugin) {
      $pluginInstance = is_string($plugin)
        ? $pluginManager->createInstance($plugin)
        : $pluginManager->createInstance($plugin['plugin'], $plugin);
      $defaultConfig = $pluginInstance instanceof ConfigurableInterface
        ? $pluginInstance->defaultConfiguration()
        : [];
      $this->drupalPlaceBlock(
        $pluginInstance->getPluginId(),
        $defaultConfig + ['weight' => $weight],
      );
    }
  }

}
