<?php

declare(strict_types=1);

namespace Drupal\Tests\babel\Traits;

use Drupal\babel\Model\Source;
use Drupal\babel\Model\StringTranslation;
use Drupal\babel\StringsCollectorFactory;

/**
 * Reusable code for strings collector testing.
 */
trait StringsCollectorTrait {

  /**
   * Asserts that a list of source strings is cached for a given language.
   *
   * @param string $langcode
   *   The language code.
   * @param list<string>|array<string, string> $strings
   *   A list of strings. Keys are sources, values are translation. If a value
   *   is NULL, that means it's an untranslated string. If the array  passed as
   *   a list (sequence), the translations are build as "[$langcode] $source".
   */
  protected function assertCached(string $langcode, array $strings): void {
    $data = $this->assertCacheExists($langcode);

    // For sake of test readability, remove the two testing languages from the
    // assertion. The languages are added at the beginning of the test but are
    // not relevant for testing.
    unset($data[(new Source('Bulgarian', ''))->getHash()]);
    unset($data[(new Source('Italian', ''))->getHash()]);

    $this->assertCount(count($strings), $data);

    // If it's a list/sequence build standard translations.
    if (array_is_list($strings)) {
      $strings = array_map(
        fn(string $source) => "[$langcode] $source",
        array_combine($strings, $strings),
      );
    }

    foreach ($strings as $source => $translation) {
      $hash = (new Source($source, ''))->getHash();
      $this->assertArrayHasKey($hash, $data);
      $string = StringTranslation::fromArray($data[$hash]);
      $this->assertSame($source, $string->source->string);
      $this->assertSame('', $string->source->context);
      $this->assertSame($translation ? $langcode : NULL, $string->getTranslation()?->language);
      $this->assertSame($translation, $string->getTranslation()?->string);
    }
  }

  /**
   * Asserts that 'cache.babel' bin contains a cache entry.
   *
   * @param string $langcode
   *   The language code.
   *
   * @return array
   *   Returns the cache entry data.
   */
  protected function assertCacheExists(string $langcode): array {
    $this->container->get(StringsCollectorFactory::class)->destruct();
    $cache = \Drupal::cache('babel')->get("strings.$langcode");
    $this->assertNotFalse($cache);
    $this->assertIsArray($cache->data);
    return $cache->data;
  }

  /**
   * Asserts that 'cache.babel' bin doesn't contain a cache entry.
   *
   * @param string $langcode
   *   The language code.
   */
  protected function assertCacheNotExists(string $langcode): void {
    $this->container->get(StringsCollectorFactory::class)->destruct();
    $this->assertFalse(\Drupal::cache('babel')->get("strings.$langcode"));
  }

}
