<?php

declare(strict_types=1);

namespace Drupal\babel_content_entity;

use Drupal\babel\BabelStorageInterface;
use Drupal\babel_content_entity\Plugin\Babel\TranslationType\ContentEntity;

/**
 * Batch operations callbacks.
 */
class BatchHelper {

  /**
   * Batch callback to process content entities.
   *
   * @param string $entityTypeId
   *   The entity type ID.
   * @param list<string|int> $ids
   *   Entity IDs.
   * @param int $total
   *   Total number of items to process. Used to for the interactive message.
   * @param array $context
   *   The batch context.
   */
  public static function processContentEntities(string $entityTypeId, array $ids, int $total, array &$context): void {
    $defaultLanguage = \Drupal::languageManager()->getDefaultLanguage();
    $context['results']['progress'] ??= 0;

    $storage = \Drupal::entityTypeManager()->getStorage($entityTypeId);
    $service = \Drupal::service(BabelContentEntityService::class);
    $entityType = \Drupal::entityTypeManager()->getDefinition($entityTypeId);

    $sources = [];
    foreach ($storage->loadMultiple($ids) as $entity) {
      // Avoid duplicate sources created from translations of entity.
      if ($entity->language()->getId() === $defaultLanguage->getId()) {
        $sources = [...$sources, ...$service->getSourcesForEntity($entity)];
      }
    }
    \Drupal::service(BabelStorageInterface::class)->update(ContentEntity::id($entityTypeId), $sources);

    $context['results']['progress'] += count($ids);
    $context['message'] = t('Processed @progress out of @total @type entities', [
      '@progress' => $context['results']['progress'],
      '@total' => $total,
      '@type' => $entityType->getSingularLabel(),
    ]);
  }

}
