<?php

declare(strict_types=1);

namespace Drupal\babel_content_entity\EventSubscriber;

use Drupal\babel\BabelStorageInterface;
use Drupal\babel\Plugin\Babel\TranslationTypePluginManager;
use Drupal\babel_content_entity\BabelContentEntityService;
use Drupal\babel_content_entity\Plugin\Babel\TranslationType\ContentEntity;
use Drupal\Core\Config\ConfigCrudEvent;
use Drupal\Core\Config\ConfigEvents;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Listens to changes to configuration.
 */
class BabelContentEntityConfigSubscriber implements EventSubscriberInterface {

  public function __construct(
    protected readonly BabelStorageInterface $babelStorage,
    protected readonly BabelContentEntityService $service,
    protected readonly TranslationTypePluginManager $translationTypeManager,
    protected readonly EntityTypeBundleInfoInterface $bundleInfo,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      ConfigEvents::SAVE => [['onSettingsSave'], ['onPluginConfigChange']],
    ];
  }

  /**
   * Updates the Babel storage when the list of entity types is changing.
   */
  public function onSettingsSave(ConfigCrudEvent $event): void {
    $config = $event->getConfig();
    if ($config->getName() !== 'babel_content_entity.settings' || !$event->isChanged('entity_type')) {
      return;
    }

    $original = $config->getOriginal('entity_type') ?? [];
    $actual = $config->get('entity_type');

    foreach (array_diff($original, $actual) as $entityTypeId) {
      $this->babelStorage->delete(ContentEntity::id($entityTypeId));
    }

    foreach (array_diff($actual, $original) as $entityTypeId) {
      $this->service->batchAddSources($entityTypeId);
    }

    // Make sure translation type plugin definitions are fetched again.
    $this->translationTypeManager->clearCachedDefinitions();
  }

  /**
   * Updates the Babel storage when the plugin configuration changes.
   */
  public function onPluginConfigChange(ConfigCrudEvent $event): void {
    $config = $event->getConfig();
    if ($config->getName() !== 'babel.settings') {
      return;
    }

    $plugins = $config->get('translation_type.plugin') ?? [];
    foreach ($plugins as $pluginId => $settings) {
      if (!str_starts_with($pluginId, ContentEntity::id('')) || !$event->isChanged("translation_type.plugin.$pluginId")) {
        continue;
      }

      [, $entityTypeId] = explode(':', $pluginId, 2);

      $allBundles = array_keys($this->bundleInfo->getBundleInfo($entityTypeId));
      $initialBundles = $config->getOriginal("translation_type.plugin.$pluginId.bundle") ?: $allBundles;
      $actualBundles = $settings['bundle'];

      // If no bundle is configured, all bundles are in play.
      $newBundles = $actualBundles ?: $allBundles;

      if ($added = array_diff($newBundles, $initialBundles)) {
        $this->service->batchAddSources($entityTypeId, $added);
      }

      foreach (array_diff($initialBundles, $newBundles) as $bundle) {
        $this->babelStorage->delete($pluginId, "$bundle:");
      }
    }

    // Make sure translation type plugin definitions are fetched again.
    $this->translationTypeManager->clearCachedDefinitions();
  }

}
