<?php

declare(strict_types=1);

namespace Drupal\babel_content_entity\Hook;

use Drupal\babel\BabelStorageInterface;
use Drupal\babel\Plugin\Babel\TranslationTypePluginManager;
use Drupal\babel_content_entity\BabelContentEntityService;
use Drupal\babel_content_entity\Plugin\Babel\TranslationType\ContentEntity;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Hook\Attribute\Hook;

/**
 * Babel Content Entity CRUD operations hooks.
 */
class BabelContentEntityHook {

  public function __construct(
    private readonly TranslationTypePluginManager $translationTypePluginManager,
    private readonly BabelContentEntityService $service,
    protected readonly BabelStorageInterface $storage,
  ) {}

  /**
   * Implements hook_entity_insert().
   *
   * @phpstan-ignore-next-line as Attribute class Drupal\Core\Hook\Attribute\Hook does not exist.
   */
  #[Hook('entity_insert')]
  public function onEntityInsert(EntityInterface $entity): void {
    $this->service->updateSourcesForEntity($entity);
  }

  /**
   * Implements hook_entity_update()
   *
   * @phpstan-ignore-next-line as Attribute class Drupal\Core\Hook\Attribute\Hook does not exist.
   */
  #[Hook('entity_update')]
  public function onEntityUpdate(EntityInterface $entity): void {
    $this->service->updateSourcesForEntity($entity);
  }

  /**
   * Implements hook_entity_delete().
   *
   * @phpstan-ignore-next-line as Attribute class Drupal\Core\Hook\Attribute\Hook does not exist.
   */
  #[Hook('entity_delete')]
  public function onEntityDelete(EntityInterface $entity): void {
    $this->storage->delete(ContentEntity::id($entity->getEntityTypeId()), "{$entity->bundle()}:{$entity->id()}:");
  }

  /**
   * Implements hook_entity_bundle_delete().
   *
   * @phpstan-ignore-next-line as Attribute class Drupal\Core\Hook\Attribute\Hook does not exist.
   */
  #[Hook('entity_bundle_delete')]
  public function onBundleDelete(string $entityTypeId, string $bundle): void {
    $pluginId = ContentEntity::id($entityTypeId);
    if (!$this->translationTypePluginManager->hasDefinition($pluginId)) {
      return;
    }

    $plugin = $this->translationTypePluginManager->createInstance($pluginId);
    $bundles = $plugin->getConfiguration()['bundle'];

    if (!$bundles || in_array($bundle, $bundles, TRUE)) {
      $this->storage->delete($pluginId, "$bundle:");
    }
  }

}
