<?php

declare(strict_types=1);

namespace Drupal\Tests\babel_content_entity\Kernel;

use Drupal\babel\Model\Source;
use Drupal\babel\Model\StringTranslation;
use Drupal\babel\Plugin\Babel\TranslationTypePluginManager;
use Drupal\babel_content_entity\Plugin\Babel\TranslationType\ContentEntity;
use Drupal\taxonomy\Entity\Vocabulary;
use Drupal\Tests\babel\Kernel\BabelKernelTestBase;
use Drupal\Tests\babel\Traits\BatchTrait;

/**
 * Tests the Babel backend update.
 *
 * @group babel
 */
class BabelContentEntityTest extends BabelKernelTestBase {

  use BatchTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'filter',
    'taxonomy',
    'text',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->container->get('module_installer')->install(['babel_content_entity']);

    $this->installEntitySchema('taxonomy_term');
    Vocabulary::create(['vid' => 'tags', 'name' => 'Tags'])->save();
    Vocabulary::create(['vid' => 'category', 'name' => 'Category'])->save();
  }

  /**
   * @covers \Drupal\babel_content_entity\Plugin\Babel\TranslationType\ContentEntity
   * @covers \Drupal\babel_content_entity\BabelContentEntityService
   * @covers \Drupal\babel_content_entity\BatchHelper
   */
  public function testBackendUpdate(): void {
    $manager = $this->container->get(TranslationTypePluginManager::class);
    $storage = \Drupal::entityTypeManager()->getStorage('taxonomy_term');

    $pluginId = ContentEntity::id('taxonomy_term');

    // No entity type has been yet configured.
    $this->assertNull($manager->getDefinition($pluginId, FALSE));

    // Create 2 terms in different vocabularies.
    $storage->create(['vid' => 'tags', 'name' => 'Tag 1', 'tid' => 1])->save();
    $storage->create(['vid' => 'category', 'name' => 'Category 1', 'tid' => 2])->save();

    // Configure taxonomy_term entity.
    $this->config('babel_content_entity.settings')
      ->set('entity_type', ['taxonomy_term'])
      ->save();

    // Updates are process in batch operation.
    $this->runPendingBatches();

    $plugin = $manager->createInstance($pluginId);
    $key = "translation_type.plugin.$pluginId.bundle";

    // Check that, on enabling the entity type, all content has been processed.
    $this->assertEquals([
      'tags:1:name:0' => new StringTranslation(
        source: new Source('Tag 1', '', TRUE)
      ),
      'category:2:name:0' => new StringTranslation(
        source: new Source('Category 1', '', TRUE)
      ),
    ], $plugin->getStrings('fr'));

    // Disable the 'category' bundle.
    $this->config('babel.settings')->set($key, ['tags'])->save();
    $this->runPendingBatches();

    $this->assertEquals([
      'tags:1:name:0' => new StringTranslation(
        source: new Source('Tag 1', '', TRUE)
      ),
    ], $plugin->getStrings('fr'));

    // Create more terms in both vocabularies.
    $storage->create(['vid' => 'tags', 'name' => 'Tag 2', 'tid' => 3])->save();
    $storage->create(['vid' => 'category', 'name' => 'Category 2', 'tid' => 4])->save();

    $this->assertEquals([
      'tags:1:name:0' => new StringTranslation(
        source: new Source('Tag 1', '', TRUE)
      ),
      'tags:3:name:0' => new StringTranslation(
        source: new Source('Tag 2', '', TRUE)
      ),
    ], $plugin->getStrings('fr'));

    // Re-enable the 'category' bundle.
    $this->config('babel.settings')->set($key, [])->save();
    $this->runPendingBatches();

    $this->assertEquals([
      'tags:1:name:0' => new StringTranslation(
        source: new Source('Tag 1', '', TRUE)
      ),
      'tags:3:name:0' => new StringTranslation(
        source: new Source('Tag 2', '', TRUE)
      ),
      'category:2:name:0' => new StringTranslation(
        source: new Source('Category 1', '', TRUE)
      ),
      'category:4:name:0' => new StringTranslation(
        source: new Source('Category 2', '', TRUE)
      ),
    ], $plugin->getStrings('fr'));

    // Delete an entity.
    $storage->load(3)->delete();

    $this->assertEquals([
      'tags:1:name:0' => new StringTranslation(
        source: new Source('Tag 1', '', TRUE)
      ),
      'category:2:name:0' => new StringTranslation(
        source: new Source('Category 1', '', TRUE)
      ),
      'category:4:name:0' => new StringTranslation(
        source: new Source('Category 2', '', TRUE)
      ),
    ], $plugin->getStrings('fr'));

    // Change the name and add a description.
    $storage->load(2)
      ->set('name', 'Category 1 (changed)')
      ->set('description', 'Nice category')
      ->save();

    $this->assertEquals([
      'tags:1:name:0' => new StringTranslation(
        source: new Source('Tag 1', '', TRUE)
      ),
      'category:2:name:0' => new StringTranslation(
        source: new Source('Category 1 (changed)', '', TRUE)
      ),
      'category:2:description:0' => new StringTranslation(
        source: new Source('Nice category', '', TRUE)
      ),
      'category:4:name:0' => new StringTranslation(
        source: new Source('Category 2', '', TRUE)
      ),
    ], $plugin->getStrings('fr'));

    // Cannot add a source string for entities in a different language.
    $storage->create([
      'vid' => 'tags',
      'name' => 'Tag 3',
      'tid' => 5,
      'langcode' => static::LANGCODE,
    ])->save();

    $this->assertEquals([
      'tags:1:name:0' => new StringTranslation(
        source: new Source('Tag 1', '', TRUE)
      ),
      'category:2:name:0' => new StringTranslation(
        source: new Source('Category 1 (changed)', '', TRUE)
      ),
      'category:2:description:0' => new StringTranslation(
        source: new Source('Nice category', '', TRUE)
      ),
      'category:4:name:0' => new StringTranslation(
        source: new Source('Category 2', '', TRUE)
      ),
    ], $plugin->getStrings('fr'));

    // Translations should not register sources.
    $term = $storage->load(2);
    $term->addTranslation(static::LANGCODE, $term->toArray())
      ->set('name', static::LANGCODE . ' Category 1')
      ->set('description', static::LANGCODE . ' Nice category')
      ->save();

    $this->assertEquals([
      'tags:1:name:0' => new StringTranslation(
        source: new Source('Tag 1', '', TRUE)
      ),
      'category:2:name:0' => new StringTranslation(
        source: new Source('Category 1 (changed)', '', TRUE)
      ),
      'category:2:description:0' => new StringTranslation(
        source: new Source('Nice category', '', TRUE)
      ),
      'category:4:name:0' => new StringTranslation(
        source: new Source('Category 2', '', TRUE)
      ),
    ], $plugin->getStrings('cz'));

    // Delete all.
    $storage->delete($storage->loadMultiple());

    // Remove the taxonomy_term entity type.
    $this->config('babel_content_entity.settings')
      ->set('entity_type', [])
      ->save();

    $this->assertNull($manager->getDefinition('content_entity:taxonomy_term', FALSE));

    // At this point the plugin is gone. Perform direct query for assertion.
    $instances = $this->container->get('database')
      ->select('babel_source_instance', 'bsi')
      ->condition('bsi.plugin', 'content_entity:taxonomy_term')
      ->countQuery()
      ->execute()
      ->fetchField();
    $this->assertEquals(0, $instances);
  }

}
