<?php

declare(strict_types=1);

namespace Drupal\Tests\babel_content_entity\Kernel;

use Drupal\babel\BabelStringsRepositoryInterface;
use Drupal\babel\StringsCollectorFactory;
use Drupal\language\Entity\ConfigurableLanguage;
use Drupal\taxonomy\Entity\Vocabulary;
use Drupal\Tests\babel\Kernel\BabelKernelTestBase;
use Drupal\Tests\babel\Traits\StringsCollectorTrait;

/**
 * @coversDefaultClass \Drupal\babel\StringsCollector
 * @group babel
 */
class StringsCollectorTest extends BabelKernelTestBase {

  use StringsCollectorTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'content_translation',
    'filter',
    'taxonomy',
    'text',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('taxonomy_term');

    $this->container->get('module_installer')->install(['babel_content_entity']);
    $this->config('babel_content_entity.settings')
      ->set('entity_type', ['taxonomy_term'])
      ->save();

    Vocabulary::create(['vid' => 'tags', 'name' => 'Tags'])->save();

    ConfigurableLanguage::createFromLangcode('bg')->save();

    $this->container->get('content_translation.manager')
      ->setEnabled('taxonomy_term', 'tags', TRUE);
  }

  /**
   * @covers \Drupal\babel_content_entity\BabelContentEntityService::updateSourcesForEntity
   */
  public function testContentEntityTranslationOutsideBabel(): void {
    $storage = $this->container->get('entity_type.manager')->getStorage('taxonomy_term');
    $repository = $this->container->get(BabelStringsRepositoryInterface::class);
    $factory = $this->container->get(StringsCollectorFactory::class);

    $term = $storage->create([
      'vid' => 'tags',
      'name' => 'Term 1',
      'description' => 'Description of Term 1',
    ]);
    $term->save();

    $translation = $term->addTranslation('bg', $term->toArray())
      ->set('name', '[bg] Term 1')
      ->set('description', '[bg] Description of Term 1');
    $translation->save();
    $factory->destruct();

    // Warm the cache.
    $repository->getStrings('bg');
    // Check cached strings.
    $this->assertCached('bg', [
      'Term 1' => '[bg] Term 1',
      'Description of Term 1' => '[bg] Description of Term 1',
    ]);

    $translation
      ->set('name', 'Updated [bg] Term 1')
      ->set('description', 'Updated [bg] Description of Term 1');
    $translation->save();
    $factory->destruct();

    // Warm the cache.
    $repository->getStrings('bg');
    // Check cached strings.
    $this->assertCached('bg', [
      'Term 1' => 'Updated [bg] Term 1',
      'Description of Term 1' => 'Updated [bg] Description of Term 1',
    ]);

    $entity = $storage->load($term->id());
    $entity->removeTranslation('bg');
    $entity->save();
    $factory->destruct();

    // Warm the cache.
    $repository->getStrings('bg');
    // Check cached strings have no translation.
    $this->assertCached('bg', [
      'Term 1' => NULL,
      'Description of Term 1' => NULL,
    ]);

  }

}
