<?php

declare(strict_types=1);

namespace Drupal\babel_menu_link_content\Hook;

use Drupal\babel\BabelStorageInterface;
use Drupal\babel\Plugin\Babel\TranslationTypePluginManager;
use Drupal\babel_menu_link_content\BabelMenuLinkContentService;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\menu_link_content\MenuLinkContentInterface;
use Drupal\system\MenuInterface;

/**
 * Babel menu link content CRUD operations hooks.
 */
class BabelMenuLinkContentHook {

  public function __construct(
    private readonly TranslationTypePluginManager $translationTypePluginManager,
    private readonly BabelMenuLinkContentService $babelMenuLinkContentService,
    protected readonly BabelStorageInterface $babelStorage,
  ) {}

  /**
   * Implements hook_ENTITY_TYPE_insert() for menu_link_content entity type.
   *
   * @phpstan-ignore-next-line as Attribute class Drupal\Core\Hook\Attribute\Hook does not exist.
   */
  #[Hook('menu_link_content_insert')]
  public function insertMenuLinkContent(MenuLinkContentInterface $link): void {
    $this->babelMenuLinkContentService->updateSourcesForLink($link);
  }

  /**
   * Implements hook_ENTITY_HOOK_update() for menu_link_content entity type.
   *
   * @phpstan-ignore-next-line as Attribute class Drupal\Core\Hook\Attribute\Hook does not exist.
   */
  #[Hook('menu_link_content_update')]
  public function updateMenuLinkContent(MenuLinkContentInterface $link): void {
    $this->babelMenuLinkContentService->updateSourcesForLink($link);

    // @todo Is it is possible to change the link menu? If yes:
    //   - Check if the new menu is allowed and if yes, add/update the entry to
    //     the {babel_source_instance} table.
    //   - If there is an entry in the {babel_source_instance} table and the new
    //     menu is not allowed, remove the entry from the
    //     {babel_source_instance} table.
  }

  /**
   * Implements hook_ENTITY_TYPE_delete() for menu_link_content entity type.
   *
   * @phpstan-ignore-next-line as Attribute class Drupal\Core\Hook\Attribute\Hook does not exist.
   */
  #[Hook('menu_link_content_delete')]
  public function deleteMenuLinkContent(MenuLinkContentInterface $link): void {
    $this->babelStorage->delete('menu_link_content', "{$link->getMenuName()}:{$link->id()}");
  }

  /**
   * Implements hook_ENTITY_TYPE_delete() for menu_link_content entity type.
   *
   * @phpstan-ignore-next-line as Attribute class Drupal\Core\Hook\Attribute\Hook does not exist.
   */
  #[Hook('menu_delete')]
  public function deleteMenuContent(MenuInterface $menu): void {
    $plugin = $this->translationTypePluginManager->createInstance('menu_link_content');
    $menus = $plugin->getConfiguration()['menu'] ?? [];

    if (!$menus || in_array($menu->id(), $menus, TRUE)) {
      $this->babelStorage->delete('menu_link_content', "{$menu->id()}:");
    }
  }

}
