<?php

declare(strict_types=1);

namespace Drupal\babel_menu_link_content\Plugin\Babel\TranslationType;

use Drupal\babel\BabelStorageInterface;
use Drupal\babel\Model\Source;
use Drupal\babel\Model\StringTranslation;
use Drupal\babel\Model\Translation;
use Drupal\babel\Plugin\Babel\TranslationType;
use Drupal\babel\Plugin\Babel\TranslationTypePluginBase;
use Drupal\babel\StringsCollectorFactory;
use Drupal\babel_menu_link_content\BabelMenuLinkContentService;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\PluginFormInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\system\MenuInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Handles translatable strings from Custom Menu Links module.
 */
#[TranslationType(
  id: 'menu_link_content',
  label: new TranslatableMarkup('Menu Link Content'),
)]
class MenuLinkContent extends TranslationTypePluginBase implements PluginFormInterface {

  public function __construct(
    array $configuration,
    string $pluginId,
    array $pluginDefinition,
    BabelStorageInterface $babelStorage,
    StringsCollectorFactory $collectorFactory,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected BabelMenuLinkContentService $batchHelper,
  ) {
    parent::__construct($configuration, $pluginId, $pluginDefinition, $babelStorage, $collectorFactory);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): self {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get(BabelStorageInterface::class),
      $container->get(StringsCollectorFactory::class),
      $container->get('entity_type.manager'),
      $container->get(BabelMenuLinkContentService::class),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return ['menu' => []] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function getStrings(string $langcode, array $ids = []): array {
    if (!$ids) {
      $ids = $this->babelStorage->getBaseQuery(
        pluginId: $this->getPluginId(),
        includeSortKey: FALSE
      )->execute()->fetchCol();
    }

    $lids = [];
    foreach ($ids as $id) {
      [, $lid, $fieldName, $delta] = explode(':', $id);
      $lids[$lid][] = [$fieldName, $delta];
    }

    $storage = $this->entityTypeManager->getStorage('menu_link_content');

    $strings = [];
    foreach ($storage->loadMultiple(array_keys($lids)) as $lid => $link) {
      $hasTranslation = $link->hasTranslation($langcode);
      if ($hasTranslation) {
        $translated = $link->getTranslation($langcode);
      }

      foreach ($lids[$lid] as [$fieldName, $delta]) {
        $source = $link?->get($fieldName)?->get($delta)->getString();

        if (!$source) {
          continue;
        }

        $translatedString = $hasTranslation
          ? $translated?->get($fieldName)?->get($delta)?->getString()
          : NULL;
        $translation = $translatedString ? new Translation(
          language: $langcode,
          string: $translatedString,
        ) : NULL;
        $id = "{$link->getMenuName()}:$lid:$fieldName:$delta";
        $strings[$id] = new StringTranslation(
          source: new Source(string: $source, context: '', status: TRUE),
          translation: $translation,
        );
      }
    }

    return $strings;
  }

  /**
   * {@inheritdoc}
   */
  public function updateTranslation(StringTranslation $string, string $id, string $langcode, string $translation): void {
    [, $lid, $fieldName, $delta] = explode(':', $id);

    $storage = $this->entityTypeManager->getStorage('menu_link_content');
    if ($link = $storage->load($lid)) {
      if ($link->hasTranslation($langcode)) {
        $translated = $link->getTranslation($langcode);
      }
      else {
        $translated = $link->addTranslation($langcode, $link->toArray());
      }

      $translated->get($fieldName)->setValue([$delta => $translation]);
      $translated->save();
    }

    parent::updateTranslation($string, $id, $langcode, $translation);
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form['menu'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Limit to menus'),
      '#description' => $this->t('Select menus to limit the translation of menu links. No selection means all menus are translatable.'),
      '#options' => array_map(
        fn(MenuInterface $menu): string => $menu->label(),
        $this->entityTypeManager->getStorage('menu')->loadMultiple(),
      ),
      '#default_value' => $this->getConfiguration()['menu'],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state): void {}

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    $all = $this->entityTypeManager->getStorage('menu')->getQuery()->execute();
    $currentMenus = $this->getConfiguration()['menu'] ?: $all;

    $configuration = $form_state->getValues();
    // Cleanup the result of the form element values.
    $configuration['menu'] = array_keys(array_filter($configuration['menu']));
    $this->setConfiguration($configuration);

    // If no menu was selected, all menus are in play.
    $newMenus = $configuration['menu'] ?: $all;

    if ($added = array_diff($newMenus, $currentMenus)) {
      $this->batchHelper->batchAddSources($added);
    }

    $removed = array_diff($currentMenus, $newMenus);
    foreach ($removed as $menu) {
      $this->babelStorage->delete('menu_link_content', "$menu:");
    }
  }

}
