<?php

declare(strict_types=1);

namespace Drupal\babel;

use Drupal\babel\Plugin\Babel\TranslationTypePluginManager;
use Drupal\Core\Database\Connection;
use Drupal\Core\Database\Query\SelectInterface;

/**
 * Storage service for babel data.
 */
class BabelStorage implements BabelStorageInterface {

  /**
   * IDs to delete on destruction.
   *
   * @var array<non-empty-string, list<string>>
   */
  protected array $idsToDelete = [];

  public function __construct(
    protected Connection $db,
    protected readonly TranslationTypePluginManager $translationTypeManager,
    protected readonly StringsCollectorFactory $collectorFactory,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function update(string $pluginId, array $sources): void {
    if (!$sources) {
      return;
    }

    $data = [];
    foreach ($sources as $id => $source) {
      $hash = $source->getHash();
      $this->db->merge('babel_source_instance')
        ->keys(['plugin' => $pluginId, 'id' => $id])
        ->fields(['hash' => $hash])
        ->execute();
      $data[$hash] = $source->getSortKey();
    }

    $existingStatuses = $this->db->select('babel_source')
      ->fields('babel_source', ['hash', 'status'])
      ->condition('hash', array_keys($data), 'IN')
      ->execute()
      ->fetchAllKeyed();

    // Invalidate the strings cache of existing hashes being updated.
    $this->collectorFactory->invalidateHashes(array_keys($existingStatuses));

    $pluginStatus = $this->translationTypeManager->getDefinition($pluginId)['status'];

    // Batch upsert babel_source records.
    foreach (array_chunk($data, 200, TRUE) as $chunk) {
      $values = [];
      foreach ($chunk as $hash => $sortKey) {
        $existingStatus = isset($existingStatuses[$hash]) ? (bool) $existingStatuses[$hash] : NULL;
        // Don't UPSERT if any of the following are true:
        // - An existing source string with the same hash and the status equals
        //   TRUE exists. Existing published strings status is preserved.
        // - There's an existing source with the same hash and the plugin's
        //   default status is FALSE, as it's a useless query with no changes.
        if ($existingStatus || (is_bool($existingStatus) && $pluginStatus === FALSE)) {
          continue;
        }

        $values[] = [
          'hash' => $hash,
          'status' => (int) $pluginStatus,
          'sort_key' => $sortKey,
        ];
      }

      if (!$values) {
        continue;
      }

      $upsert = $this->db->upsert('babel_source')
        ->fields(['hash', 'status', 'sort_key'])
        ->key('hash');
      foreach ($values as $value) {
        $upsert->values($value);
      }
      $upsert->execute();
    }
  }

  /**
   * {@inheritdoc}
   */
  public function delete(string $pluginId, ?string $startsWith = NULL, ?array $ids = NULL): void {
    if ($startsWith !== NULL && $ids !== NULL) {
      throw new \InvalidArgumentException('The $startsWith and $ids are mutually exclusive. Only one of them can be passed.');
    }

    $select = $this->db->select('babel_source_instance')
      ->distinct()
      ->fields('babel_source_instance', ['hash'])
      ->condition('plugin', $pluginId);

    $delete = $this->db->delete('babel_source_instance')
      ->condition('plugin', $pluginId);

    if ($startsWith) {
      $select->condition('id', $this->db->escapeLike($startsWith) . '%', 'LIKE');
      $delete->condition('id', $this->db->escapeLike($startsWith) . '%', 'LIKE');
    }
    elseif (is_array($ids)) {
      if (!$ids) {
        // Distinguish between passing NULL (don't use this filter) and empty
        // array (delete none).
        return;
      }

      $select->condition('id', $ids, 'IN');
      $delete->condition('id', $ids, 'IN');
    }

    // Invalidate the strings cache of existing hashes being deleted.
    $this->collectorFactory->invalidateHashes($select->execute()->fetchCol());

    $delete->execute();
    $this->updateSourceTable();
  }

  /**
   * {@inheritdoc}
   */
  public function scheduleDeleteById(string $pluginId, array $ids): void {
    $this->idsToDelete[$pluginId] = array_merge(
      $this->idsToDelete[$pluginId] ?? [],
      array_values($ids)
    );
  }

  /**
   * {@inheritdoc}
   */
  public function destruct(): void {
    foreach ($this->idsToDelete as $pluginId => $ids) {
      $this->delete(pluginId: $pluginId, ids: $ids);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function updateStatusForHash(string $hash, bool $status): void {
    $this->db->update('babel_source')
      ->fields(['status' => (int) $status])
      ->condition('hash', $hash)
      ->execute();
  }

  /**
   * {@inheritdoc}
   */
  public function getStatusForHash(string $hash): bool {
    return (bool) $this->db->select('babel_source', 'bs')
      ->fields('bs', ['status'])
      ->condition('hash', $hash)
      ->execute()
      ->fetchField();
  }

  /**
   * {@inheritdoc}
   */
  public function getSourceStringInstances(string $hash): array {
    $query = $this->db->select('babel_source_instance')
      ->fields('babel_source_instance', ['plugin', 'id'])
      ->condition('hash', $hash);

    $instances = [];
    foreach ($query->execute() as $row) {
      $instances[$row->plugin][] = $row->id;
    }

    return $instances;
  }

  /**
   * {@inheritdoc}
   */
  public function hashExists(string $hash): bool {
    $query = $this->db->select('babel_source_instance', 'bsi')
      ->fields('bsi', ['hash'])
      ->condition('bs.hash', $hash);
    // Check both tables: A hash that exists only in one of them, is not valid.
    $query->innerJoin('babel_source', 'bs', 'bsi.hash = bs.hash');

    return (bool) $query->execute()->fetchField();
  }

  /**
   * {@inheritdoc}
   */
  public function getBaseQuery(string $pluginId, array $limitToIds = [], array $fields = ['id'], bool $includeSortKey = TRUE): SelectInterface {
    $query = $this->db->select('babel_source_instance', 'bsi')
      ->fields('bsi', $fields)
      ->condition('bsi.plugin', $pluginId);

    if ($limitToIds) {
      $query->condition('bsi.id', $limitToIds, 'IN');
    }

    $query->innerJoin('babel_source', 'bs', 'bsi.hash = bs.hash');
    if ($includeSortKey) {
      $query->addField('bs', 'sort_key');
    }

    return $query;
  }

  /**
   * Updates {babel_source} table.
   */
  protected function updateSourceTable(): void {
    $this->db->query("DELETE bs FROM {babel_source} bs LEFT JOIN {babel_source_instance} bsi ON bs.hash = bsi.hash WHERE bsi.hash IS NULL ");
  }

}
