<?php

declare(strict_types=1);

namespace Drupal\babel\EventSubscriber;

use Drupal\babel\BabelStorageInterface;
use Drupal\babel\StringsCollectorFactory;
use Drupal\locale\LocaleEvent;
use Drupal\locale\LocaleEvents;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Listens to locale events.
 */
class LocaleSubscriber implements EventSubscriberInterface {

  /**
   * List of event signatures used to avoid duplicate processing.
   *
   * @var list<string>
   */
  protected array $signatures = [];

  public function __construct(
    protected readonly BabelStorageInterface $babelStorage,
    protected readonly StringsCollectorFactory $collectorFactory,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [LocaleEvents::SAVE_TRANSLATION => 'onTranslationSave'];
  }

  /**
   * Invalidates the translation cache.
   *
   * @param \Drupal\locale\LocaleEvent $event
   *   The event.
   */
  public function onTranslationSave(LocaleEvent $event): void {
    // This event might be dispatched several times in the same request with the
    // same values. Make sure we're only processing once.
    if ($this->alreadyProcessed($event)) {
      return;
    }

    $lids = $event->getLids();

    // If no specific locale IDs are provided, we don't need to invalidate
    // specific hashes. The event can be dispatched with empty lids when
    // clearing all caches for affected languages.
    if (empty($lids)) {
      return;
    }

    $hashes = $this->babelStorage->getBaseQuery(pluginId: 'locale', fields: ['hash'])
      ->condition('id', $lids, 'IN')
      ->execute()
      ->fetchCol();

    if (!$hashes) {
      return;
    }

    foreach ($event->getLangCodes() as $langcode) {
      $this->collectorFactory->invalidateHashes($hashes, $langcode);
    }
  }

  /**
   * Checks whether this event has been already processed.
   *
   * @param \Drupal\locale\LocaleEvent $event
   *   The event.
   *
   * @return bool
   *   Whether this event has been already processed.
   */
  protected function alreadyProcessed(LocaleEvent $event): bool {
    $lids = $event->getLids();
    $langcodes = $event->getLangCodes();

    // Make it predictable.
    sort($lids);
    sort($langcodes);

    $signature = serialize([$langcodes, $lids]);
    $alreadyVisited = in_array($signature, $this->signatures, TRUE);

    $this->signatures[] = $signature;

    return $alreadyVisited;
  }

}
