<?php

declare(strict_types=1);

namespace Drupal\babel\Form;

use Drupal\babel\Plugin\Babel\DataTransferPluginInterface;
use Drupal\babel\Plugin\Babel\DataTransferPluginManager;
use Drupal\babel\Plugin\Babel\TranslationTypePluginInterface;
use Drupal\babel\Plugin\Babel\TranslationTypePluginManager;
use Drupal\Component\Transliteration\TransliterationInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\DependencyInjection\AutowireTrait;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\ConfigTarget;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Form\SubformState;
use Drupal\Core\Plugin\PluginFormInterface;
use Drupal\Core\StreamWrapper\StreamWrapperManagerInterface;
use Symfony\Component\DependencyInjection\Attribute\Autowire;

/**
 * Babel settings form.
 */
class BabelSettingsForm extends ConfigFormBase {

  use AutowireTrait;

  public function __construct(
    ConfigFactoryInterface $configFactory,
    TypedConfigManagerInterface $typedConfigManager,
    protected TranslationTypePluginManager $translationTypeManager,
    protected DataTransferPluginManager $dataTransferManager,
    #[Autowire(service: 'transliteration')]
    protected TransliterationInterface $transliteration,
    protected StreamWrapperManagerInterface $streamManager,
  ) {
    parent::__construct($configFactory, $typedConfigManager);
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildForm($form, $form_state);

    foreach ($this->getPluginTypes() as $key => $info) {
      $form[$key] = [
        '#type' => 'details',
        '#tree' => TRUE,
        '#title' => $info['title'],
        '#description' => $info['description'],
        '#open' => TRUE,
        'plugin' => [],
      ];

      foreach ($info['manager']->getDefinitions() as $pluginId => $definition) {
        $plugin = $info['manager']->createInstance($pluginId);
        assert($plugin instanceof $info['interface']);

        $form[$key]['plugin'][$pluginId] = [
          '#type' => 'details',
          '#title' => $definition['label'],
        ];

        if (is_a($definition['class'], PluginFormInterface::class, TRUE)) {
          $subFormState = SubformState::createForSubform($form[$key]['plugin'][$pluginId], $form, $form_state);
          $form[$key]['plugin'][$pluginId] += $plugin->buildConfigurationForm($form[$key]['plugin'][$pluginId], $subFormState);
        }
        else {
          $form[$key]['plugin'][$pluginId]['help'] = [
            '#markup' => $this->t('The %label @type plugin is not configurable', [
              '%label' => $definition['label'],
              '@type' => $info['label'],
            ]),
          ];
        }
      }
    }

    $form['data_transfer']['destination'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Exported file destination'),
      '#description' => $this->t('Should be a directory referred by stream wrapper. Use private:// for a higher level of privacy.'),
      '#required' => TRUE,
      '#config_target' => 'babel.settings:data_transfer.destination',
    ];

    $form['data_transfer']['prefix'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Exported file prefix'),
      '#description' => $this->t('If provided the file name pattern is [PREFIX]-[LANGCODE].[EXT], otherwise [LANGCODE].[EXT]. The value is transliterated.'),
      '#maxlength' => 25,
      '#config_target' => new ConfigTarget(
        configName: 'babel.settings',
        propertyPath: 'data_transfer.prefix',
        toConfig: [$this, 'sanitizeFileNamePrefix'],
      ),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    parent::validateForm($form, $form_state);

    foreach ($this->getPluginTypes() as $key => $info) {
      if (!$form_state->hasValue([$key, 'plugin'])) {
        continue;
      }

      foreach ($form_state->getValue([$key, 'plugin']) as $pluginId => $configuration) {
        $plugin = $info['manager']->createInstance($pluginId);
        assert($plugin instanceof $info['interface']);

        if ($plugin instanceof PluginFormInterface) {
          $subFormState = SubformState::createForSubform($form[$key]['plugin'][$pluginId], $form, $form_state);
          $plugin->validateConfigurationForm($form[$key]['plugin'][$pluginId], $subFormState);
        }
      }
    }

    if (!$scheme = $this->streamManager->getScheme($form_state->getValue(['data_transfer', 'destination']))) {
      $form_state->setErrorByName('data_transfer][destination', $this->t('Invalid scheme'));
    }
    if (!$this->streamManager->isValidScheme($scheme)) {
      $form_state->setErrorByName('data_transfer][destination', $this->t('Invalid scheme %scheme', ['%scheme' => $scheme]));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    foreach ($this->getPluginTypes() as $key => $info) {
      foreach ($info['manager']->getDefinitions() as $pluginId => $definition) {
        if ($form_state->hasValue([$key, 'plugin', $pluginId])) {
          $plugin = $info['manager']->createInstance($pluginId);
          assert($plugin instanceof $info['interface']);

          $subFormState = SubformState::createForSubform($form[$key]['plugin'][$pluginId], $form, $form_state);
          $plugin->submitConfigurationForm($form[$key]['plugin'][$pluginId], $subFormState);
          $subFormState->setValues($plugin->getConfiguration());
        }
        else {
          $form_state->setValue([$key, 'plugin', $pluginId], []);
        }

        // Make use of #config_target mechanism.
        $map = $form_state->get(static::CONFIG_KEY_TO_FORM_ELEMENT_MAP) ?? [];
        $map['babel.settings']["$key.plugin.$pluginId"] = [$key, 'plugin', $pluginId];
        $form_state->set(static::CONFIG_KEY_TO_FORM_ELEMENT_MAP, $map);
        $form[$key]['plugin'][$pluginId]['#config_target'] = "babel.settings:$key.plugin.$pluginId";
      }
    }

    parent::submitForm($form, $form_state);
  }

  /**
   * Returns the list of the plugin types.
   *
   * @return array<array-key, array{title: \Drupal\Core\StringTranslation\TranslatableMarkup, description: \Drupal\Core\StringTranslation\TranslatableMarkup, label: \Drupal\Core\StringTranslation\TranslatableMarkup, manager: \Drupal\Core\Plugin\DefaultPluginManager, interface: \Drupal\Component\Plugin\PluginInspectionInterface|\Drupal\Component\Plugin\ConfigurableInterface}>
   *   List of plugin types.
   */
  protected function getPluginTypes(): array {
    return [
      'translation_type' => [
        'title' => $this->t('Translation type plugin configurations'),
        'description' => $this->t('Configuration of plugins responsible to harvest translatable strings and save their translations.'),
        'label' => $this->t('translation type'),
        'manager' => $this->translationTypeManager,
        'interface' => TranslationTypePluginInterface::class,
      ],
      'data_transfer' => [
        'title' => $this->t('Data transfer (import/export) settings'),
        'description' => $this->t('Configuration of plugins responsible to import and/or export translations.'),
        'label' => $this->t('data transfer'),
        'manager' => $this->dataTransferManager,
        'interface' => DataTransferPluginInterface::class,
      ],
    ];
  }

  /**
   * Sanitizes a file name prefix.
   *
   * @param string $prefix
   *   The prefix to sanitize.
   *
   * @return string
   *   The sanitized prefix.
   */
  public function sanitizeFileNamePrefix(string $prefix): string {
    if (!$prefix = trim($prefix)) {
      return $prefix;
    }
    $initial = $prefix;

    // Transliterate.
    $prefix = $this->transliteration->transliterate($prefix, 'en', '-');

    // Replace whitespace.
    $prefix = preg_replace('/\s/u', '-', trim($prefix));

    // Replace non-alphanumeric.
    $prefix = preg_replace('/[^0-9A-Za-z_.-]/u', '-', $prefix);

    // Replace multiple separators with single one.
    $prefix = preg_replace('/(_)_+|(\.)\.+|(-)-+/u', '-', $prefix);
    $prefix = preg_replace('/(_|\.|\-)[(_|\.|\-)]+/u', '-', $prefix);
    $prefix = preg_replace('/' . preg_quote('-') . '[' . preg_quote('-') . ']*/u', '-', $prefix);

    // Remove replacement character from the end.
    $prefix = mb_strtolower(trim($prefix, '-'));

    if ($prefix !== $initial) {
      $this->messenger()->addStatus($this->t('The file prefix %initial has been sanitized as %prefix.', [
        '%initial' => $initial,
        '%prefix' => $prefix,
      ]));
    }

    return $prefix;
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['babel.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'babel_settings_form';
  }

}
