<?php

declare(strict_types=1);

namespace Drupal\babel\Model;

use Drupal\Component\Transliteration\PhpTransliteration;

/**
 * Represents a source string.
 */
class Source {

  /**
   * The hash of the source.
   *
   * @var string
   */
  protected string $hash;

  /**
   * The value to use when sorting source objects.
   *
   * @var string
   */
  protected string $sortKey;

  public function __construct(
    public readonly string $string,
    public readonly string $context,
    protected ?bool $status = NULL,
  ) {}

  /**
   * Sets the given status.
   *
   * @param bool $status
   *   The status to set.
   *
   * @return $this
   */
  public function setStatus(bool $status): self {
    $this->status = $status;
    return $this;
  }

  /**
   * Returns the status of this string translation object.
   *
   * @return bool
   *   The current status.
   */
  public function getStatus(): bool {
    return $this->status;
  }

  /**
   * Toggles the status.
   *
   * @return $this
   */
  public function toggleStatus(): self {
    if (is_bool($this->status)) {
      $this->status = !$this->status;
    }
    return $this;
  }

  /**
   * Returns a unique identifier for the source string.
   *
   * @return string
   *   Hash identifying a source string.
   */
  public function getHash(): string {
    if (!isset($this->hash)) {
      $this->hash = hash('sha256', "$this->string:$this->context");
    }
    return $this->hash;
  }

  /**
   * Returns a key that allows a human relevant sorting of source strings.
   *
   * @return string
   *   Sorting key.
   */
  public function getSortKey(): string {
    if (!isset($this->sortKey)) {
      $this->sortKey = strip_tags($this->string) . ':' . $this->context;

      // Replace chars such as \n, \t, \r with space.
      $this->sortKey = preg_replace('/\s/', ' ', $this->sortKey);

      // Replace consecutive spaces with one space.
      $this->sortKey = preg_replace('/\s{2,}/', ' ', $this->sortKey);

      // Transliterate.
      $transliterator = new PhpTransliteration();
      $this->sortKey = $transliterator->transliterate($this->sortKey);

      // Remove any non-alphanumeric sequence from the start of the string, but
      // only if the string contains alphanumeric characters.
      $this->sortKey = preg_replace('#^[^a-z0-9]+([a-z0-9].*)$#i', '$1', $this->sortKey);

      // 64 characters limit is enough to show a decent sorting for humans.
      return substr($this->sortKey, 0, 64);
    }

    return $this->sortKey;
  }

  /**
   * Returns the source array representation.
   *
   * @return array{string: string, context: string, status: bool|null}
   *   The source array representation.
   */
  public function toArray(): array {
    $return = [
      'string' => $this->string,
      'context' => $this->context,
    ];
    if ($this->status !== NULL) {
      $return['status'] = $this->status;
    }
    return $return;
  }

  /**
   * Creates a source instance out of an array.
   *
   * @param array{source: string, context: string, status?: bool|null} $values
   *   The input array.
   *
   * @return $this
   */
  public static function fromArray(array $values): self {
    return new static($values['string'], $values['context'], $values['status'] ?? NULL);
  }

}
