<?php

declare(strict_types=1);

namespace Drupal\babel\Plugin\Babel;

use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Plugin\DefaultPluginManager;

/**
 * Data transfer plugin manager.
 */
class DataTransferPluginManager extends DefaultPluginManager {

  /**
   * Instantiated plugins static cache.
   *
   * @var array<array-key, \Drupal\babel\Plugin\Babel\ExporterPluginInterface>
   */
  protected array $instances = [];

  public function __construct(
    \Traversable $namespaces,
    CacheBackendInterface $cacheBackend,
    ModuleHandlerInterface $moduleHandler,
    protected readonly ConfigFactoryInterface $configFactory,
  ) {
    parent::__construct('Plugin/Babel/DataTransfer', $namespaces, $moduleHandler, DataTransferPluginInterface::class, DataTransfer::class);
    $this->alterInfo('babel_data_transfer_info');
    $this->setCacheBackend($cacheBackend, 'babel_data_transfer_info');
  }

  /**
   * {@inheritdoc}
   */
  public function createInstance($plugin_id, array $configuration = []): ExporterPluginInterface {
    if ($configuration) {
      // Normally we don't pass the plugin configuration because we solve it
      // here, see the few lines below. But if we do explicitly, we want a fresh
      // instantiation, so we reset the static cache.
      unset($this->instances[$plugin_id]);
    }

    if (!isset($this->instances[$plugin_id])) {
      $configuration = $configuration ?: $this->configFactory->get('babel.settings')->get("data_transfer.plugin.$plugin_id") ?: [];
      $this->instances[$plugin_id] = parent::createInstance($plugin_id, $configuration);
    }
    return $this->instances[$plugin_id];
  }

  /**
   * Returns the definitions of data transfer plugins acting as exporters.
   *
   * @return array<array-key, array>
   *   List of definitions keyed by plugin ID.
   */
  public function getExporters(): array {
    return array_filter(
      $this->getDefinitions(),
      fn(array $definition): bool => is_a($definition['class'], ExporterPluginInterface::class, TRUE),
    );
  }

  /**
   * Returns the definitions of data transfer plugins acting as importers.
   *
   * @return array<array-key, array>
   *   List of definitions keyed by plugin ID.
   */
  public function getImporters(): array {
    return array_filter(
      $this->getDefinitions(),
      fn(array $definition): bool => is_a($definition['class'], ImporterPluginInterface::class, TRUE),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function clearCachedDefinitions(): void {
    $this->instances = [];
    parent::clearCachedDefinitions();
  }

}
