<?php

declare(strict_types=1);

namespace Drupal\babel\Plugin\Babel;

use Drupal\babel\BabelStorageInterface;
use Drupal\babel\Model\StringTranslation;
use Drupal\babel\Model\Translation;
use Drupal\babel\StringsCollectorFactory;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Plugin\PluginBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Base class for translation type plugins.
 */
abstract class TranslationTypePluginBase extends PluginBase implements TranslationTypePluginInterface, ContainerFactoryPluginInterface {

  public function __construct(
    array $configuration,
    string $pluginId,
    array $pluginDefinition,
    protected readonly BabelStorageInterface $babelStorage,
    protected readonly StringsCollectorFactory $collectorFactory,
  ) {
    parent::__construct($configuration, $pluginId, $pluginDefinition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): self {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get(BabelStorageInterface::class),
      $container->get(StringsCollectorFactory::class),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function updateTranslation(StringTranslation $string, string $id, string $langcode, string $translation): void {
    $string->setTranslation($translation === ''
      ? NULL
      : new Translation(language: $langcode, string: $translation)
    );

    // Invalidate the cache for this string.
    $this->collectorFactory->invalidateHashes([$string->source->getHash()], $langcode);
  }

  /**
   * {@inheritdoc}
   */
  public function getConfiguration(): array {
    return $this->configuration + $this->defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function setConfiguration(array $configuration): void {
    $this->configuration = $configuration;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [];
  }

}
