<?php

declare(strict_types=1);

namespace Drupal\babel;

use Drupal\babel\Plugin\Babel\TranslationTypePluginManager;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Cache\CacheCollector;
use Drupal\Core\Lock\LockBackendInterface;
use Symfony\Component\DependencyInjection\Attribute\Autowire;

/**
 * Cache collector for translated strings.
 */
class StringsCollector extends CacheCollector {

  /**
   * Static cache for strings provided by a plugin in a certain language.
   */
  protected array $stringsByPluginAndLang = [];

  public function __construct(
    protected readonly string $langcode,
    #[Autowire(service: 'cache.babel')]
    CacheBackendInterface $cache,
    #[Autowire(service: 'lock')]
    LockBackendInterface $lock,
    protected readonly TranslationTypePluginManager $manager,
  ) {
    parent::__construct("strings.$this->langcode", $cache, $lock);
  }

  /**
   * {@inheritdoc}
   */
  protected function resolveCacheMiss($key): ?array {
    $value = NULL;

    // We cannot inject this service because of service circular reference.
    // @phpstan-ignore-next-line as \Drupal calls should be avoided in classes, use dependency injection instead
    foreach (\Drupal::service(BabelStorageInterface::class)->getSourceStringInstances($key) as $pluginId => $ids) {
      foreach ($this->getPluginStringsForLanguage($pluginId, $this->langcode, $ids) as $string) {
        // A translated string takes precedence.
        if (!$value || $string->isTranslated()) {
          // Store as array of scalars to avoid serializing class instances.
          $value = $string->toArray();
        }
      }
    }

    // Don't store malformed values.
    if ($value !== NULL) {
      $this->storage[$key] = $value;
      $this->persist($key);
    }

    return $value;
  }

  /**
   * {@inheritdoc}
   */
  public function destruct(): void {
    parent::destruct();

    // Don't keep an empty cache entry.
    if (empty($this->storage)) {
      $this->clear();
    }
  }

  /**
   * Returns the strings for a given plugin and language.
   *
   * @param string $pluginId
   *   The plugin ID.
   * @param string $langcode
   *   The language code.
   * @param list<string> $ids
   *   A list of IDs to limit the results.
   *
   * @return array<non-empty-string, \Drupal\babel\Model\StringTranslation>
   *   Associative array of string translation objects keyed by their ID.
   */
  protected function getPluginStringsForLanguage(string $pluginId, string $langcode, array $ids): array {
    if (!isset($this->stringsByPluginAndLang[$pluginId][$langcode])) {
      $plugin = $this->manager->createInstance($pluginId);

      // For the sake of performance we get all plugin's strings once and cache
      // them to take advantage of some plugins ability to perform a single
      // query instead of multiple fragmented queries.
      $this->stringsByPluginAndLang[$pluginId][$langcode] = $plugin->getStrings($this->langcode);
    }
    return array_intersect_key($this->stringsByPluginAndLang[$pluginId][$langcode], array_flip($ids));
  }

}
