<?php

declare(strict_types=1);

namespace Drupal\babel;

use Drupal\babel\Plugin\Babel\TranslationTypePluginManager;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\DestructableInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Lock\LockBackendInterface;
use Symfony\Component\DependencyInjection\Attribute\Autowire;

/**
 * Factory for translated string collectors.
 */
class StringsCollectorFactory implements DestructableInterface {

  /**
   * Static cache of string collectors.
   *
   * @var list<\Drupal\babel\StringsCollector>
   */
  protected array $collectors = [];

  public function __construct(
    #[Autowire(service: 'cache.babel')]
    protected readonly CacheBackendInterface $cache,
    #[Autowire(service: 'lock')]
    protected readonly LockBackendInterface $lock,
    protected readonly TranslationTypePluginManager $manager,
    protected readonly LanguageManagerInterface $languageManager,
  ) {}

  /**
   * Produces a new string cache collector.
   *
   * @param string $langcode
   *   The language code.
   *
   * @return \Drupal\babel\StringsCollector
   *   The string collector service.
   */
  public function get(string $langcode): StringsCollector {
    if (!isset($this->collectors[$langcode])) {
      $this->collectors[$langcode] = new StringsCollector(
        langcode: $langcode,
        cache: $this->cache,
        lock: $this->lock,
        manager: $this->manager,
      );
    }
    return $this->collectors[$langcode];
  }

  /**
   * Invalidates cached strings for a given list of hashes and language.
   *
   * @param list<string> $hashes
   *   Hashes to be invalidated.
   * @param string|null $langcode
   *   (optional) The language code for which the hash is to be invalidated. If
   *   omitted, the hash will be invalidated for all configurable languages.
   */
  public function invalidateHashes(array $hashes, ?string $langcode = NULL): void {
    if (!$hashes) {
      return;
    }

    $langcodes = $langcode ? [$langcode] : array_keys($this->languageManager->getLanguages());
    foreach ($langcodes as $langcode) {
      foreach ($hashes as $hash) {
        $this->get($langcode)->delete($hash);
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function destruct(): void {
    foreach ($this->collectors as $collector) {
      $collector->destruct();
    }

    // We're done. Reset collector list.
    $this->collectors = [];
  }

}
