<?php

declare(strict_types=1);

namespace Drupal\Tests\babel\FunctionalJavascript;

use Drupal\Core\Url;

/**
 * Tests the History API integration for bookmarkable URLs.
 *
 * @group babel
 */
class HistoryApiIntegrationTest extends BabelFunctionalJavascriptTestBase {

  /**
   * Tests URL updates when filters are applied.
   */
  public function testUrlUpdatesWithFilters(): void {
    $this->drupalLogin($this->testAccount);
    $this->drupalGet(Url::fromRoute('babel.ui'));

    $page = $this->getSession()->getPage();
    $assertSession = $this->assertSession();

    // Apply search filter and verify URL is updated.
    $page->fillField('Search', 'test');
    $page->pressButton('Filter');
    $assertSession->assertWaitOnAjaxRequest();

    $currentUrl = $this->getSession()->getCurrentUrl();
    $this->assertStringContainsString('search=test', $currentUrl);
  }

  /**
   * Tests URL parameter persistence after page load.
   */
  public function testUrlParameterPersistence(): void {
    $this->drupalLogin($this->testAccount);

    // Visit URL with parameters.
    $url = Url::fromRoute('babel.ui', [], [
      'query' => [
        'langcode' => 'de',
        'search' => 'admin',
      ],
    ]);

    $this->drupalGet($url);
    $assertSession = $this->assertSession();

    // Verify form fields are populated from URL parameters.
    $assertSession->fieldValueEquals('Language', 'de');
    $assertSession->fieldValueEquals('Search', 'admin');
  }

  /**
   * Tests empty parameters are cleaned from URL.
   */
  public function testUrlParameterCleaning(): void {
    $this->drupalLogin($this->testAccount);
    $this->drupalGet(Url::fromRoute('babel.ui'));

    $page = $this->getSession()->getPage();
    $assertSession = $this->assertSession();

    // Apply then clear search filter.
    $page->fillField('Search', 'test');
    $page->pressButton('Filter');
    $assertSession->assertWaitOnAjaxRequest();

    $page->fillField('Search', '');
    $page->pressButton('Filter');
    $assertSession->assertWaitOnAjaxRequest();

    // Verify empty parameter is removed.
    $currentUrl = $this->getSession()->getCurrentUrl();
    $this->assertStringNotContainsString('search=', $currentUrl);
  }

}
