<?php

declare(strict_types=1);

namespace Drupal\Tests\babel\Kernel\Plugin\Babel\TranslationType;

use Drupal\babel\Model\StringTranslation;
use Drupal\babel\Plugin\Babel\TranslationTypePluginManager;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\locale\LocaleTranslation;

/**
 * Tests the "locale" translation type plugin.
 *
 * @group babel
 *
 * @coversDefaultClass \Drupal\babel\Plugin\Babel\TranslationType\Locale
 */
class LocaleTest extends TranslationTypePluginTestBase {

  /**
   * {@inheritdoc}
   */
  public function setUp(): void {
    parent::setUp();

    $translation = $this->container->get('string_translator.locale.lookup');
    $this->assertInstanceOf(LocaleTranslation::class, $translation);
    // Ensure that the \Drupal\locale\LocaleTranslation::$translations property
    // has some cached translations in it. Without this, serialization will not
    // actually be tested fully.
    $translation->getStringTranslation('pt', 'Foo', '');
    $translation->getStringTranslation('pt', 'Bar baz', 'not foo');
    $translation->destruct();

    $this->plugin = $this->container
      ->get(TranslationTypePluginManager::class)
      ->createInstance('locale');
  }

  /**
   * {@inheritdoc}
   *
   * @param string $langcode
   *   The language code to test the loading with.
   * @param array $stringIds
   *   The string IDs to load.
   * @param array $expected
   *   The expected result.
   *
   * @dataProvider providerTestGetMultiple
   */
  public function testGetStrings(
    string $langcode = '',
    array $stringIds = [],
    array $expected = [],
  ): void {
    $this->assertEquals(
      $expected,
      array_map(
        fn (StringTranslation $translation) => $translation->toArray(),
        $this->plugin->getStrings($langcode, $stringIds),
      ),
    );
  }

  /**
   * Test data provider for ::testGetStrings.
   *
   * @return array[]
   *   The test cases.
   */
  public static function providerTestGetMultiple(): array {
    return [
      'Load everything' => [
        'langcode' => 'fi',
        'stringIds' => [],
        'expected' => [
          1 => [
            'source' => [
              'string' => 'Foo',
              'context' => '',
              'status' => TRUE,
            ],
          ],
          2 => [
            'source' => [
              'string' => 'Bar baz',
              'context' => 'not foo',
              'status' => TRUE,
            ],
          ],
        ],
      ],
      'Load specific IDs' => [
        'langcode' => 'be',
        'stringIds' => ['2'],
        'expected' => [
          2 => [
            'source' => [
              'string' => 'Bar baz',
              'context' => 'not foo',
              'status' => TRUE,
            ],
          ],
        ],
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function testUpdateTranslation(): void {
    $options = [
      'langcode' => $this->language->getId(),
      'context' => 'not foo',
    ];
    // Unfortunately, the sniff does not know named params.
    // phpcs:ignore Drupal.Semantics.FunctionT.NotLiteralString
    $markupInitial = new TranslatableMarkup(string: 'Bar baz', options: $options);

    // Check object before translation was added.
    $this->assertEquals('Bar baz', (string) $markupInitial);

    // Add translation.
    $string = $this->plugin->getStrings('pt')['2'];
    $this->plugin->updateTranslation(
      $string,
      '2',
      $this->language->getId(),
      '[TRANSLATED] Bar baz',
    );

    // We have to call this reset just because we are inside a Kernel test's
    // single request. In normal circumstances, the translation will be
    // available in the next request after the new translation was added.
    \Drupal::service('string_translation')->reset();

    // Check the result.
    // phpcs:ignore Drupal.Semantics.FunctionT.NotLiteralString
    $markup = new TranslatableMarkup(string: 'Bar baz', options: $options);
    $this->assertEquals('[TRANSLATED] Bar baz', (string) $markup);
  }

}
