<?php

declare(strict_types=1);

namespace Drupal\Tests\babel\Kernel;

use Drupal\babel\BabelStringsRepositoryInterface;
use Drupal\babel\EventSubscriber\ConfigSubscriber;
use Drupal\babel\Model\Source;
use Drupal\babel\Model\StringTranslation;
use Drupal\babel\Plugin\Babel\TranslationTypePluginManager;
use Drupal\babel\StringsCollectorFactory;
use Drupal\Core\Cache\Cache;
use Drupal\KernelTests\KernelTestBase;
use Drupal\language\Entity\ConfigurableLanguage;
use Drupal\Tests\babel\Traits\StringsCollectorTrait;

/**
 * @coversDefaultClass \Drupal\babel\StringsCollector
 * @group babel
 */
class StringsCollectorTest extends KernelTestBase {

  use StringsCollectorTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'babel',
    'locale',
    'language',
    'system',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    ConfigurableLanguage::createFromLangcode('bg')->save();
    ConfigurableLanguage::createFromLangcode('it')->save();

    $this->installSchema('locale', ['locales_source', 'locales_target', 'locales_location']);
    $this->installSchema('babel', ['babel_source', 'babel_source_instance']);

    $lookup = $this->container->get('string_translator.locale.lookup');
    $storage = $this->container->get('locale.storage');

    $sources = ['Foo', 'Bar', 'Baz'];
    foreach ($sources as $source) {
      $lookup->getStringTranslation('bg', $source, '');
    }
    $lookup->destruct();

    foreach (['bg', 'it'] as $langcode) {
      foreach ($sources as $source) {
        $storage->createTranslation([
          'lid' => $storage->findString(['source' => $source])->lid,
          'language' => $langcode,
          'translation' => "[$langcode] $source",
        ])->save();
      }
    }
    Cache::invalidateTags(['locale']);
  }

  /**
   * @covers ::resolveCacheMiss
   * @covers \Drupal\babel\StringsCollectorFactory::invalidateHashes
   */
  public function testStringsCollectorFactory(): void {
    $repository = $this->container->get(BabelStringsRepositoryInterface::class);
    $factory = $this->container->get(StringsCollectorFactory::class);

    $this->assertCacheNotExists('bg');
    $this->assertCacheNotExists('it');

    $repository->getStrings('bg');
    $this->assertCached('bg', ['Foo', 'Bar', 'Baz']);
    $this->assertCacheNotExists('it');

    $repository->getStrings('it');
    $this->assertCached('it', ['Foo', 'Bar', 'Baz']);

    // Invalidate for a specific language.
    $factory->invalidateHashes([(new Source('Foo', ''))->getHash()], 'bg');
    $this->assertCached('bg', ['Bar', 'Baz']);
    $this->assertCached('it', ['Foo', 'Bar', 'Baz']);

    // Invalidate for all languages.
    $factory->invalidateHashes([(new Source('Baz', ''))->getHash()]);
    $this->assertCached('bg', ['Bar']);
    $this->assertCached('it', ['Foo', 'Bar']);

    // On request, the cache is repopulated.
    $repository->getStrings('bg');
    $repository->getStrings('it');
    $this->assertCached('bg', ['Foo', 'Bar', 'Baz']);
    $this->assertCached('it', ['Foo', 'Bar', 'Baz']);
  }

  /**
   * @covers \Drupal\babel\BabelStorage::update
   * @covers \Drupal\babel\BabelStorage::delete
   */
  public function testBabelStorageCacheInvalidation(): void {
    $repository = $this->container->get(BabelStringsRepositoryInterface::class);

    // Add a translated config.
    $this->assertStringUpdate(new Source('Qux', ''));

    // Change the config.
    $this->assertStringUpdate(new Source('Waldo', ''));

    // Deleting a string, updates the cache.
    $this->config('system.site')->delete();
    foreach (['bg', 'it'] as $langcode) {
      $this->assertCached($langcode, ['Foo', 'Bar', 'Baz']);
      $strings = $repository->getStrings($langcode);
      $this->assertArrayNotHasKey((new Source('Waldo', ''))->getHash(), $strings);
    }
  }

  /**
   * @covers \Drupal\babel\Plugin\Babel\TranslationTypePluginBase::updateTranslation
   */
  public function testUpdateTranslationInvalidation(): void {
    $repository = $this->container->get(BabelStringsRepositoryInterface::class);
    $manager = $this->container->get(TranslationTypePluginManager::class);
    $storage = $this->container->get('locale.storage');
    $factory = $this->container->get(StringsCollectorFactory::class);

    $plugin = $manager->createInstance('locale');

    // Update the string's Bulgarian translation.
    $string = StringTranslation::fromArray([
      'source' => ['string' => 'Foo', 'context' => ''],
    ]);
    $lid = (string) $storage->findString(['source' => 'Foo'])->lid;
    $plugin->updateTranslation($string, $lid, 'bg', 'Updated [bg] Foo');
    // Trigger invalidation.
    $factory->destruct();
    // Warm the cache.
    $repository->getStrings('bg');
    // Check that cache has been updated.
    $this->assertCached('bg', [
      'Foo' => 'Updated [bg] Foo',
      'Bar' => '[bg] Bar',
      'Baz' => '[bg] Baz',
    ]);

    // Italian strings are not yet cached.
    $this->assertCacheNotExists('it');

    // Warm the Italian strings cache.
    $repository->getStrings('it');
    $this->assertCached('it', ['Foo', 'Bar', 'Baz']);

    // Update the string's Bulgarian translation.
    $string = StringTranslation::fromArray([
      'source' => ['string' => 'Baz', 'context' => ''],
    ]);
    $lid = (string) $storage->findString(['source' => 'Baz'])->lid;
    $plugin->updateTranslation($string, $lid, 'it', 'Updated [it] Baz');
    // Trigger invalidation.
    $factory->destruct();
    // Warm the cache.
    $repository->getStrings('it');
    // Check that cache has been updated.
    $this->assertCached('it', [
      'Foo' => '[it] Foo',
      'Bar' => '[it] Bar',
      'Baz' => 'Updated [it] Baz',
    ]);
  }

  /**
   * @covers \Drupal\babel\EventSubscriber\LocaleSubscriber::onTranslationSave
   */
  public function testLocaleTranslationOutsideBabel(): void {
    $repository = $this->container->get(BabelStringsRepositoryInterface::class);
    $localeStorage = $this->container->get('locale.storage');
    $factory = $this->container->get(StringsCollectorFactory::class);

    // Warm the cache.
    $repository->getStrings('it');
    $this->assertCached('it', ['Foo', 'Bar', 'Baz']);

    $lid = (string) $localeStorage->findString(['source' => 'Bar'])->lid;
    $localeStorage->createTranslation([
      'lid' => $lid,
      'language' => 'it',
      'translation' => "Updated [it] Bar",
    ])->save();
    // When Locale updates a translation, either via UI or by a batch process,
    // it always calls this function.
    // @see \Drupal\locale\Form\TranslateEditForm::submitForm()
    // @see locale_translate_batch_refresh()
    _locale_refresh_translations(['it'], [$lid]);
    $factory->destruct();

    // Warm the cache.
    $repository->getStrings('it');
    // Check that cache has been updated.
    $this->assertCached('it', [
      'Foo' => '[it] Foo',
      'Bar' => 'Updated [it] Bar',
      'Baz' => '[it] Baz',
    ]);
  }

  /**
   * @covers \Drupal\babel\EventSubscriber\ConfigSubscriber::onOverrideChange
   */
  public function testConfigTranslationOutsideBabel(): void {
    $repository = $this->container->get(BabelStringsRepositoryInterface::class);
    $languageManager = $this->container->get('language_manager');
    $subscriber = $this->container->get(ConfigSubscriber::class);
    $factory = $this->container->get(StringsCollectorFactory::class);

    $this->assertStringUpdate(new Source('Qux', ''));

    // Update the translation using core.
    $languageManager->getLanguageConfigOverride('bg', 'system.site')
      ->set('name', "Updated [bg] Qux")
      ->save();

    // Factory destruction always run last.
    $subscriber->destruct();
    $factory->destruct();

    // Warm the cache.
    $repository->getStrings('bg');
    $this->assertCached('bg', [
      'Foo' => '[bg] Foo',
      'Bar' => '[bg] Bar',
      'Baz' => '[bg] Baz',
      'Qux' => 'Updated [bg] Qux',
    ]);

    // Delete a translation using core.
    $languageManager->getLanguageConfigOverride('bg', 'system.site')->delete();
    $subscriber->destruct();
    $factory->destruct();
    // The string has been cached without a translation.
    $repository->getStrings('bg');
    $this->assertCached('bg', [
      'Foo' => '[bg] Foo',
      'Bar' => '[bg] Bar',
      'Baz' => '[bg] Baz',
      'Qux' => NULL,
    ]);
  }

  /**
   * Asserts that inserting/updating a string will create the correct cache.
   *
   * @param \Drupal\babel\Model\Source $source
   *   The source string being inserted/updated.
   */
  public function assertStringUpdate(Source $source): void {
    $repository = $this->container->get(BabelStringsRepositoryInterface::class);
    $languageManager = $this->container->get('language_manager');

    $this->config('system.site')->set('name', $source->string)->save();
    foreach (['bg', 'it'] as $langcode) {
      $languageManager->getLanguageConfigOverride($langcode, 'system.site')
        ->set('name', "[$langcode] $source->string")
        ->save();
    }
    $this->container->get(ConfigSubscriber::class)->destruct();

    // Test that is correctly cached and retrieved correctly from the cache.
    foreach (['bg', 'it'] as $langcode) {
      $strings = $repository->getStrings($langcode);
      $this->assertCached($langcode, ['Foo', 'Bar', 'Baz', $source->string]);
      $this->assertSame(
        "[$langcode] $source->string",
        $strings[$source->getHash()]->getTranslation()->string
      );
    }
  }

}
