<?php

declare(strict_types=1);

namespace Drupal\Tests\babel\Traits;

use Behat\Mink\Element\NodeElement;

/**
 * Trait for translating strings on the UI.
 */
trait TranslationAssertionTrait {

  /**
   * Add translation for a visible string on the translation form.
   *
   * @param string $sourceString
   *   The source string. Might be one of the (singular / plural) variants.
   * @param string $translation
   *   The translation of the source string.
   * @param int|null $pluralVersion
   *   Which variant to add the translation to? If not specified and both the
   *   source and the destination have the same amount of plural forms, this can
   *   be NULL. In this case, the plural version is automatically calculated
   *   based on the source string.
   * @param bool $save
   *   Whether the translation should be saved.
   */
  protected function translateSourceStringTo(string $sourceString, string $translation, ?int $pluralVersion = NULL, $save = TRUE): void {
    $page = $this->getSession()->getPage();
    // Find the matching source string.
    $sourceSelector = '*[@data-babel-source-variant]';
    $sourceStringItem = $page->find('xpath', "//{$sourceSelector}[contains(.,'$sourceString')]");

    $this->assertInstanceOf(NodeElement::class, $sourceStringItem);

    $sourceVariant = $sourceStringItem->getAttribute('data-babel-source-variant');
    $totalSourceVariants = count($sourceStringItem->findAll('xpath', "/ancestor::td//$sourceSelector"));
    $translationRow = $sourceStringItem->find('xpath', '/ancestor::tr');
    $translationVariants = count($translationRow->findAll('xpath', '//*[@data-babel-translation-variant]'));

    // The given string might be one of the plural forms.
    if ($totalSourceVariants !== $translationVariants) {
      $this->assertNotNull($pluralVersion);
    }
    $pluralVersion ??= (int) $sourceVariant;

    $translationInput = $sourceStringItem->find('xpath', "/ancestor::tr//*[@data-babel-translation-variant='$pluralVersion']")->find('css', 'input,textarea');
    $this->assertInstanceOf(NodeElement::class, $translationInput);
    $translationInput->setValue($translation);

    if ($save) {
      $translationRow->find('xpath', '//*[@data-babel-save]')->click();
      if (method_exists($this->assertSession(), 'assertWaitOnAjaxRequest')) {
        $this->assertSession()->assertWaitOnAjaxRequest();
      }
    }
  }

}
