<?php

namespace Drupal\background_image;

use Drupal\background_image\Plugin\ContextReaction\BackgroundImage as BackgroundImageAlias;
use Drupal\Component\Utility\Color;
use Drupal\Component\Utility\Html;
use Drupal\context\ContextManager;
use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Cache\CacheTagsInvalidator;
use Drupal\Core\Cache\CacheTagsInvalidatorInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\DependencyInjection\DependencySerializationTrait;
use Drupal\Core\Entity\EntityRepository;
use Drupal\Core\Entity\EntityRepositoryInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Image\ImageFactory;
use Drupal\Core\Image\ImageInterface;
use Drupal\Core\Routing\ResettableStackedRouteMatchInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Routing\UrlGeneratorInterface;
use Drupal\Core\State\StateInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\file\FileInterface;
use Drupal\imagemagick\Plugin\ImageToolkit\ImagemagickToolkit;
use Drupal\system\Plugin\ImageToolkit\GDToolkit;
use Drupal\webform\WebformInterface;
use Symfony\Component\DependencyInjection\ContainerAwareTrait;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a manager for background images.
 */
class BackgroundImageManager implements BackgroundImageManagerInterface {

  use DependencySerializationTrait;
  use StringTranslationTrait;

  /**
   * The configuration object for the background images.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected $config;

  /**
   * The entity storage service.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected $storage;

  /**
   * The entity view builder service.
   *
   * @var \Drupal\Core\Entity\EntityViewBuilderInterface
   */
  protected $viewBuilder;

  /**
   * BackgroundImageManager constructor.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The Config Factory service.
   * @param \Drupal\Core\Entity\EntityRepositoryInterface $entityRepository
   *   The entity repository.
   * @param \Drupal\Core\Entity\EntityTypeBundleInfoInterface $entityTypeBundleInfo
   *   The Entity Type Bundle Info service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The Entity Type manager service.
   * @param \Drupal\Core\File\FileSystemInterface $fileSystem
   *   The File System service.
   * @param \Drupal\Core\Image\ImageFactory $imageFactory
   *   The Image Factory service.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *   The Module Handler service.
   * @param \Drupal\Core\Routing\ResettableStackedRouteMatchInterface $routeMatch
   *   The Route Match service.
   * @param \Drupal\Core\State\StateInterface $state
   *   The State service.
   * @param \Drupal\Core\Routing\UrlGeneratorInterface $urlGenerator
   *   The UrlGenerator service.
   * @param \Drupal\Core\Cache\CacheTagsInvalidatorInterface $cacheTagsInvalidator
   *   The cache tags invalidator service.
   * @param \Drupal\context\ContextManager $contextManager
   *    The context manager service.
   *
   * @noinspection PhpDocMissingThrowsInspection
   */
  public function __construct(
    protected ConfigFactoryInterface $configFactory,
    protected EntityRepositoryInterface $entityRepository,
    protected EntityTypeBundleInfoInterface $entityTypeBundleInfo,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected FileSystemInterface $fileSystem,
    protected ImageFactory $imageFactory,
    protected ModuleHandlerInterface $moduleHandler,
    protected RouteMatchInterface $routeMatch,
    protected StateInterface $state,
    protected UrlGeneratorInterface $urlGenerator,
    protected CacheTagsInvalidatorInterface $cacheTagsInvalidator,
    protected ContextManager $contextManager,
  ) {
    $this->config = $this->configFactory->get('background_image.settings');
    $this->storage = $this->entityTypeManager->getStorage('media');
    $this->viewBuilder = $this->entityTypeManager->getViewBuilder('media');
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new self(
      $container->get('config.factory'),
      $container->get('entity.repository'),
      $container->get('entity_type.bundle.info'),
      $container->get('entity_type.manager'),
      $container->get('file_system'),
      $container->get('image.factory'),
      $container->get('module_handler'),
      $container->get('current_route_match'),
      $container->get('state'),
      $container->get('url_generator.non_bubbling'),
      $container->get('cache_tags.invalidator'),
      $container->get('context.manager'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function alterEntityForm(array &$form, FormStateInterface $form_state) {
    // Check if inline_entity_form exists.
    $inline_entity_form = $this->moduleHandler->moduleExists('inline_entity_form');

    // Only alter forms that have a "inline_entity_form_entity" set.
    // @see \Drupal\background_image\BackgroundImageManager::prepareForm
    /** @var \Drupal\Core\Entity\EntityInterface $entity */
    $entity = $form_state->get('inline_entity_form_entity');
    if (!$inline_entity_form || !$entity) {
      return;
    }

    $group = $this->getEntityConfig($entity, 'group');
    $require = $this->getEntityConfig($entity, 'require');
    $background_image = $this->getEntityBackgroundImage($entity);

    $form['background_image'] = [
      '#type' => 'details',
      '#theme_wrappers' => ['details__background_image'],
      '#title' => $this->t('Background Image'),
      '#open' => !$require && $group ? FALSE : TRUE,
      '#group' => $group,
      '#weight' => $group ? NULL : 100,
      '#tree' => TRUE,
    ];

    $form['background_image']['inline_entity_form'] = [
      '#theme_wrappers' => NULL,
      '#type' => 'inline_entity_form',
      '#entity_type' => 'background_image',
      '#langcode' => $entity->language()->getId(),
      '#default_value' => $background_image,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function colorIsDark($hex = NULL) {
    if (!isset($hex)) {
      return FALSE;
    }
    $rgb = array_values(Color::hexToRgb($hex));
    return (0.213 * $rgb[0] + 0.715 * $rgb[1] + 0.072 * $rgb[2] < 255 / 2);
  }

  /**
   * {@inheritdoc}
   */
  public function colorSampleFile(?FileInterface $file = NULL, $default = NULL) {
    return isset($file) ? $this->colorSampleImage($this->imageFactory->get($file->getFileUri()), $default) : $default;
  }

  /**
   * {@inheritdoc}
   */
  public function colorSampleImage(ImageInterface $image, $default = NULL) {
    // Immediately return if the image is not valid.
    if (!$image->isValid()) {
      return $default;
    }

    // Retrieve the toolkit and use it, if valid.
    $toolkit = $image->getToolkit();
    if ($toolkit instanceof GDToolkit) {
      return $this->colorSampleGdImage($toolkit, $default);
    }
    elseif ($toolkit instanceof ImagemagickToolkit) {
      return $this->colorSampleImagemagickImage($toolkit, $default);
    }
    return $default;
  }

  /**
   * Determines the average color of an image using the GD toolkit.
   *
   * @param \Drupal\system\Plugin\ImageToolkit\GDToolkit $image
   *   A GD image toolkit object.
   * @param string $default
   *   A default lowercase simple color (HEX) representation to use if
   *   unable to sample the image.
   *
   * @return string
   *   An associative array with red, green, blue and alpha keys that contains
   *   the appropriate values for the specified color index.
   */
  protected function colorSampleGdImage(GDToolkit $image, $default = NULL) {
    if ($image->apply('resize', ['width' => 1, 'height' => 1]) && ($resource = $image->getImage())) {
      return @Color::rgbToHex(array_slice(@imagecolorsforindex($resource, @imagecolorat($resource, 0, 0)), 0, 3)) ?: $default;
    }
    return $default;
  }

  /**
   * Determines the average color of an image using the Imagemagick toolkit.
   *
   * Due to how Imagemagick's toolkit works in Drupal, this doesn't actually
   * use any of the methods provided by the toolkit. This is because it
   * operates under the assumption that the output will be saved as an image.
   *
   * Since this is using an external binary and requires reading the text
   * output, the arguments must be constructed manually and the
   * ImagemagickExecManager service must be used directly.
   *
   * @param \Drupal\imagemagick\Plugin\ImageToolkit\ImagemagickToolkit $image
   *   An Imagemagick toolkit object.
   * @param string $default
   *   A default lowercase simple color (HEX) representation to use if
   *   unable to sample the image.
   *
   * @return string
   *   An associative array with red, green, blue and alpha keys that contains
   *   the appropriate values for the specified color index.
   *
   * @noinspection PhpDocMissingThrowsInspection
   * @see https://stackoverflow.com/a/25488429
   *
   */
  protected function colorSampleImagemagickImage(ImagemagickToolkit $image, $default = NULL) {
    // Note: this service cannot be injected because not everyone will have
    // this module installed. It can only be accessed here via runtime.
    /** @var \Drupal\imagemagick\ImagemagickExecManagerInterface $exec_manager */
    $arguments = (new \ReflectionClass('\Drupal\imagemagick\ImagemagickExecArguments'))->newInstance($this->execManager)
      ->setSourceLocalPath($this->fileSystem->realpath($image->getSource()));

    $arguments->add('-resize 1x1\!')
      ->add('-format "%[fx:int(255*r+.5)],%[fx:int(255*g+.5)],%[fx:int(255*b+.5)]"')
      ->add('info:-');

    if ($this->execManager->execute('convert', $arguments, $output, $error)) {
      return @Color::rgbToHex(explode(',', $output)) ?: $default;
    }

    return $default;
  }

  /**
   * {@inheritdoc}
   */
  public function getBackgroundImage($langcode = NULL, array $context = [], $cid = '') {
    $activeReactions = $this->contextManager->getActiveReactions('background_image');
    $activeReaction = current($activeReactions);
    $media = $activeReaction ? $activeReaction->execute() : NULL;
    $definition = $this->entityTypeManager->getDefinition('context');
    $cacheability = ($media ? CacheableMetadata::createFromObject($media) : new CacheableMetadata())
      ->addCacheTags($definition->getListCacheTags())
      ->addCacheContexts($definition->getListCacheContexts());

    foreach ($this->contextManager->getContexts() as $context) {
      foreach ($context->getReactions() as $reaction) {
        // Add cacheability for contexts that have background reactions.
        if ($reaction instanceof BackgroundImageAlias) {
          $cacheability->addCacheContexts(["active_contexts:{$context->id()}"]);
          $cacheability->addCacheableDependency($context);
        }
        // And a cache tag we can invalidate if a background reaction is added
        // to an existing context.
        $cacheability->addCacheTags(['context.reaction.background']);
      }
    }
    return [$media, $cacheability];
  }

  /**
   * {@inheritdoc}
   */
  public function getBaseClass() {
    return Html::cleanCssIdentifier(preg_replace('/^(\.|#)/', '', $this->config->get('css.base_class') ?: 'background-image'));
  }

  /**
   * {@inheritdoc}
   */
  public static function service() {
    return \Drupal::service('background_image.manager');
  }

  /**
   * {@inheritdoc}
   */
  public function useMinifiedCssUri() {
    return $this->configFactory->get('system.performance')->get('css.preprocess') && $this->cssMinifier;
  }

  /**
   * {@inheritdoc}
   */
  public function view($background_image, $view_mode = 'full', $langcode = NULL) {
    return $this->viewBuilder->view($background_image, $view_mode, $langcode);
  }

}
