<?php

namespace Drupal\background_image\Entity;

use Drupal\background_image\BackgroundImageInterface;
use Drupal\Component\Render\FormattableMarkup;
use Drupal\Core\Entity\ContentEntityBase;
use Drupal\Core\Entity\EntityChangedTrait;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\Link;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Url;
use Drupal\user\EntityOwnerTrait;

/**
 * Defines the Background Image entity.
 *
 * @ingroup background_image
 *
 * @ContentEntityType(
 *   id = "background_image",
 *   label = @Translation("Background Image"),
 *   label_collection = @Translation("Background Image"),
 *   label_singular = @Translation("background image"),
 *   label_plural = @Translation("background images"),
 *   label_count = @PluralTranslation(
 *     singular = "@count background image",
 *     plural = "@count background images"
 *   ),
 *   handlers = {
 *     "access" = "Drupal\background_image\BackgroundImageAccessControlHandler",
 *     "list_builder" = "Drupal\background_image\BackgroundImageListBuilder",
 *     "translation" = "Drupal\content_translation\ContentTranslationHandler",
 *     "views_data" = "Drupal\views\EntityViewsData",
 *     "form" = {
 *       "add" = "Drupal\background_image\Form\BackgroundImageHandlerForm",
 *       "edit" = "Drupal\background_image\Form\BackgroundImageHandlerForm",
 *       "delete" = "Drupal\background_image\Form\BackgroundImageDeleteForm",
 *     },
 *   },
 *   admin_permission = "administer background image",
 *   base_table = "background_image",
 *   data_table = "background_image_field_data",
 *   translatable = TRUE,
 *   entity_keys = {
 *     "id" = "bid",
 *     "label" = "label",
 *     "uuid" = "uuid",
 *     "langcode" = "langcode",
 *   },
 *   links = {
 *     "canonical" = "/admin/config/media/background_image/{background_image}/edit",
 *     "edit-form" = "/admin/config/media/background_image/{background_image}/edit",
 *     "delete-form" = "/admin/config/media/background_image/{background_image}/delete",
 *     "collection" = "/admin/config/media/background_image"
 *   },
 * )
 */
class BackgroundImage extends ContentEntityBase implements BackgroundImageInterface {

  use EntityChangedTrait;
  use EntityOwnerTrait;
  use StringTranslationTrait;

  /**
   * The EntityRepositoryInterface for interacting with entities.
   *
   * @var \Drupal\Core\Entity\EntityRepositoryInterface
   */
  protected $entityRepository;

  /**
   * A hash representing the settings configuration for background images.
   *
   * @var string
   */
  protected $settingsHash;

  /**
   * A hash representing the image associated with the background.
   *
   * @var string
   */
  protected $imageHash;

  /**
   * The BackgroundImageSettings object containing configuration settings.
   *
   * @var \Drupal\background_image\BackgroundImageSettings
   */
  protected $settings;

  /**
   * The parent BackgroundImageInterface, if applicable.
   *
   * @var \Drupal\background_image\BackgroundImageInterface|null
   */
  protected $parent;

  /**
   * {@inheritdoc}
   */
  public function __sleep(): array {
    // Don't use unset() here because the magic method
    // \Drupal\Core\Entity\ContentEntityBase::__get can cause the value to be
    // set as a FieldItemList object. Instead, always explicitly set to NULL.
    $this->cssSelector = NULL;
    $this->settingsHash = NULL;
    $this->imageHash = NULL;
    $this->parent = NULL;
    $this->settings = NULL;
    return parent::__sleep();
  }

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type) {
    /** @var \Drupal\Core\Field\BaseFieldDefinition[] $fields */
    $fields = parent::baseFieldDefinitions($entity_type);

    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(t('Authored on'))
      ->setDescription(t('The time that the background image was created.'))
      ->setDisplayOptions('view', [
        'region' => 'hidden',
      ]);

    $fields['changed'] = BaseFieldDefinition::create('changed')
      ->setLabel(t('Changed'))
      ->setDescription(t('The time that the background image was last edited.'))
      ->setDisplayOptions('view', [
        'region' => 'hidden',
      ]);

    $fields['image'] = BaseFieldDefinition::create('image')
      ->setLabel(t('Image'))
      ->setDescription(t('The background image to display.'))
      ->setSettings([
        'file_directory' => 'background_image',
        'alt_field' => 0,
        'alt_field_required' => 0,
        'title_field' => 0,
        'title_field_required' => 0,
        'max_resolution' => '',
        'min_resolution' => '',
        'default_image' => [
          'uuid' => NULL,
          'alt' => '',
          'title' => '',
          'width' => NULL,
          'height' => NULL,
        ],
      ])
      ->setDisplayOptions('form', [
        'type' => 'image_image',
        'weight' => 2,
      ])
      ->setDisplayOptions('view', [
        'label' => 'hidden',
        'weight' => 0,
      ])
      ->setTranslatable(TRUE);

    $fields['media'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Media Image'))
      ->setDescription(t('An entity reference to the media used for the background image or video.'))
      ->addConstraint('ReferenceAccess')
      ->addConstraint('ValidReference')
      ->setRequired(TRUE)
      ->setSettings([
        'target_type' => 'media',
      ]);

    $fields['label'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Label'))
      ->setDescription(t('An administrative description to help identify this specific background image.'))
      ->setRequired(TRUE)
      ->setTranslatable(TRUE)
      ->setDefaultValue('');

    $fields['settings'] = BaseFieldDefinition::create('map')
      ->setLabel(t('Settings'))
      ->setDescription(t('Specific settings for this background image.'))
      ->setTranslatable(TRUE);

    $fields['type'] = BaseFieldDefinition::create('list_integer')
      ->setLabel(t('Type'))
      ->setDescription(t('Choose when this background image should be displayed.'))
      ->setRequired(TRUE)
      ->setDefaultValue(self::TYPE_GLOBAL)
      ->setSettings([
        'allowed_values' => self::getTypes(),
      ]);

    $fields['target'] = BaseFieldDefinition::create('string_long')
      ->setLabel(t('Target'))
      ->setDescription(t('A target, if any.'))
      ->setDefaultValue('');

    $fields['uid'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('User'))
      ->setDescription(t('An entity reference to the user who created this background image.'))
      ->addConstraint('ReferenceAccess')
      ->addConstraint('ValidReference')
      ->setRequired(TRUE)
      ->setSettings([
        'target_type' => 'user',
      ]);

    return $fields;
  }

  /**
   * Retrieves the Entity Repository service.
   *
   * @return \Drupal\Core\Entity\EntityRepositoryInterface|mixed
   *   The entity repository service.
   */
  protected function getEntityRepository() {
    if (!isset($this->entityRepository)) {
      $this->entityRepository = $this->getContainer()->get('entity.repository');
    }
    return $this->entityRepository;
  }

  /**
   * {@inheritdoc}
   */
  public function getTarget($explode = FALSE) {
    $target = $this->get('target')->value;
    return $target && $explode ? explode(':', $target) : $target;
  }

  /**
   * Returns the currently set langcode for the entity.
   *
   * @todo Remove if https://www.drupal.org/project/drupal/issues/2303877 lands.
   *
   * @return string
   *   The currently set langcode.
   */
  public function getLanguageCode() {
    if ($this->activeLangcode !== LanguageInterface::LANGCODE_DEFAULT) {
      return $this->activeLangcode;
    }
    return $this->defaultLangcode;
  }

  /**
   * {@inheritdoc}
   */
  public static function getTypes() {
    return [
      self::TYPE_GLOBAL => t('Global'),
      self::TYPE_ENTITY => t('Entity'),
      self::TYPE_ENTITY_BUNDLE => t('Entity Bundle'),
      // @todo Re-enable once support has been properly done.
      // self::TYPE_PATH => t('Path'),
      // self::TYPE_ROUTE => t('Route'),
      self::TYPE_VIEW => t('View'),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getTargetEntity($type = NULL, $target = NULL, $langcode = NULL, array $context = []) {
    if (!isset($type)) {
      $type = $this->getType();
    }
    if (!isset($target)) {
      $target = $this->getTarget();
    }
    $entity = NULL;
    if ($type === self::TYPE_ENTITY && isset($target)) {
      [$entity_type, $entity_id] = explode(':', $target);
      if (isset($entity_type) && isset($entity_id)) {
        $entity = $this->getEntityRepository()->loadEntityByUuid($entity_type, $entity_id);
      }
    }

    // Load entity with translation context.
    if ($entity instanceof EntityInterface) {
      return $this->getEntityRepository()
        ->getTranslationFromContext($entity, $langcode ?: $this->getLanguageCode(), $context);
    }

    return $entity;
  }

  /**
   * {@inheritdoc}
   */
  public function getTargetEntityBundle($type = NULL, $target = NULL) {
    if (!isset($type)) {
      $type = $this->getType();
    }
    if (!isset($target)) {
      $target = $this->getTarget();
    }
    $entity = NULL;
    if ($type === self::TYPE_ENTITY_BUNDLE && $target) {
      [$entity_type_id, $entity_bundle] = explode(':', $target);
      if (isset($entity_type_id) && isset($entity_bundle) && ($entity_type = $this->entityTypeManager()
          ->getDefinition($entity_type_id))) {
        if ($bundle_entity_type = $entity_type->getBundleEntityType()) {
          $entity = $this->entityTypeManager()->getStorage($bundle_entity_type)->load($entity_bundle);
        }
        else {
          $entity = $entity_type;
        }
      }
    }
    return $entity;
  }

  /**
   * {@inheritdoc}
   */
  public function getTargetView($type = NULL, $target = NULL) {
    if (!isset($type)) {
      $type = $this->getType();
    }
    if (!isset($target)) {
      $target = $this->getTarget();
    }
    /** @var \Drupal\views\ViewEntityInterface $view */
    $view = NULL;
    if ($type === self::TYPE_VIEW) {
      [$view_id, $display_id] = explode(':', $target);

      if (isset($view_id) && isset($display_id) && ($view = $this->entityTypeManager()
          ->getStorage('view')
          ->load($view_id))) {
        $view_executable = $view->getExecutable();
        $view_executable->setDisplay($display_id);
        if (!$this->getBackgroundImageManager()->validView($view)) {
          $view = NULL;
        }
      }
    }
    return $view;
  }

  /**
   * {@inheritdoc}
   */
  public function getTargetWebform($type = NULL, $target = NULL) {
    if (!isset($type)) {
      $type = $this->getType();
    }
    if (!isset($target)) {
      $target = $this->getTarget();
    }
    /** @var \Drupal\page_manager\PageInterface $page */
    $webform = NULL;
    if ($target && $type === self::TYPE_ENTITY) {
      [$entity_type, $entity_type_id] = explode(':', $target);
      if ($webform = $this->entityTypeManager()->getStorage($entity_type)->load($entity_type_id)) {
        if (!$this->getBackgroundImageManager()->validEntity($webform)) {
          $webform = NULL;
        }
      }
    }

    return $webform;
  }

  /**
   * {@inheritdoc}
   */
  public function getType() {
    return (int) $this->get('type')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function getTypeLabel($link = FALSE) {
    if (!isset($type)) {
      $type = $this->getType();
    }

    $types = self::getTypes();
    if (!isset($types[$type])) {
      return $this->t('Unknown');
    }

    if ($label = $this->label($link)) {
      if ($type === self::TYPE_ENTITY || $type === self::TYPE_ENTITY_BUNDLE || $type === self::TYPE_VIEW) {
        return $label;
      }
      return new FormattableMarkup('@type: @label',
        ['@type' => $types[$type], '@label' => $label]
      );
    }

    return $types[$type];
  }

  /**
   * {@inheritdoc}
   */
  public function label($link = FALSE) {
    if ($entity = $this->getTargetEntity()) {
      return $this->t('%entity_type (@entity_id): @entity_label', [
        '%entity_type' => $this->getBackgroundImageManager()->getEntityBundleLabel($entity) ?: $entity->getEntityType()
          ->getLabel(),
        '@entity_label' => $link ? $entity->toLink()->toString() : $entity->label(),
        '@entity_id' => $entity->id(),
      ]);
    }
    elseif ($entity_bundle = $this->getTargetEntityBundle()) {
      if ($entity_bundle instanceof EntityInterface) {
        return $this->t('%entity_type (@entity_id): @entity_label', [
          '%entity_type' => $entity_bundle->getEntityType()->getLabel(),
          '@entity_label' => $link && $entity_bundle->hasLinkTemplate('edit-form') ? $entity_bundle->toLink(NULL, 'edit-form')
            ->toString() : $entity_bundle->label(),
          '@entity_id' => $entity_bundle->id(),
        ]);
      }
      elseif ($entity_bundle instanceof EntityTypeInterface) {
        return $this->t('%entity_type (@entity_id)', [
          '%entity_type' => $entity_bundle->getLabel(),
          '@entity_id' => $entity_bundle->id(),
        ]);
      }
    }
    elseif ($view = $this->getTargetView()) {
      $executable = $view->getExecutable();
      $display = $executable->getDisplay();
      $path = FALSE;
      if ($display->hasPath()) {
        $path = '/' . $display->getPath();
        if ($view->status() && strpos($path, '%') === FALSE) {
          $path = Link::fromTextAndUrl($path, Url::fromUserInput($path));
        }
      }
      return $this->t('View (@entity_id): @entity_label', [
        '@entity_label' => $link && $path ? $path->toString() : $view->label(),
        '@entity_id' => "{$view->id()}:{$executable->current_display}",
      ]);
    }
    elseif ($webform = $this->getTargetWebform()) {
      return $this->t('Webform (@entity_id): @entity_label', [
        '@entity_label' => $link && $webform->hasLinkTemplate('edit-form') ? $webform->toLink(NULL, 'edit-form')
          ->toString() : $webform->label(),
        '@entity_id' => "{$webform->id()}",
      ]);
    }
    $label = parent::label();
    return isset($label) ? trim($label) : NULL;
  }

}
