<?php

declare(strict_types=1);

namespace Drupal\background_image\Form;

use Drupal\Core\Cache\CacheTagsInvalidator;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configure Background Image settings for this site.
 */
final class BackgroundImageSettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'background_image_background_image_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['background_image.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $form['introduction'] = [
      '#type' => 'markup',
      '#markup' => $this->t('<h2>Background image rewritten</h2>
<p>The background image module has been completely rewritten. It now uses the Media module and
related support (eg responsive image styles, image style processing) to provide the bulk of the
image processing and rendering support. A new "background image" media type should be used to upload
images and styles should then be configured to control breakpoints that are used, and to apply
any transformations such as resizing, bluring or the like. Settings that apply to a particular
background image can be found on the form for adding / editing individual background images
(media items).</p>
<p>To decide which image is displayed on a particular page, the Context module is used, with
reactions to a context now including the ability to choose a background image.</p>
'),
    ];
    $form['base_class'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Base class'),
      '#default_value' => $this->config('background_image.settings')->get('base_class') ?? 'background_image',
      '#description' => 'The prefix used for classes that the module generates for styling images',
    ];
    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $this->config('background_image.settings')
      ->set('base_class', $form_state->getValue('base_class'))
      ->save();
    parent::submitForm($form, $form_state);
    CacheTagsInvalidator::invalidateTags($form['#base_class']);
  }

}
