<?php

namespace Drupal\background_image\Plugin\Block;

use Drupal\background_image\BackgroundImageManager;
use Drupal\Core\Block\BlockBase;
use Drupal\Core\Cache\Cache;
use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Menu\LocalTaskManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a background image block.
 *
 * @Block(
 *   id = "background_image",
 *   admin_label = @Translation("Background Image"),
 *   category = @Translation("Background Image")
 * )
 */
class BackgroundImageBlock extends BlockBase implements ContainerFactoryPluginInterface {

  /**
   * Creates a BackgroundImageBlock instance.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin ID for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager service.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, protected EntityTypeManagerInterface $entityTypeManager) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function baseConfigurationDefaults() {
    $defaults = parent::baseConfigurationDefaults();
    $defaults['label_display'] = 'hidden';
    return $defaults;
  }

  /**
   * {@inheritdoc}
   */
  public function build($cid = '') {
    $background_image_manager = BackgroundImageManager::service();
    [$media, $cacheability] = $background_image_manager->getBackgroundImage(NULL, [], $cid);
    $build = $media ? $background_image_manager->view($media, 'image') : [];
    $cacheability->addCacheableDependency(CacheableMetadata::createFromRenderArray($build))
      ->applyTo($build);
    if ($media) {
      $build['#attached']['library'][] = 'background_image/background_image';
    }
    // We want the block rendered even if we have no background image, so that
    // if that changes, the refresh token can do its work.
    $build['#attributes'] = [];
    return $build;
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form['label_display']['#access'] = FALSE;

    return $form;
  }

}
