<?php

namespace Drupal\background_image;

use Drupal\background_image\Entity\BackgroundImage;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Image\ImageInterface;
use Drupal\file\FileInterface;
use Drupal\views\ViewEntityInterface;
use Drupal\webform\WebformInterface;
use Symfony\Component\DependencyInjection\ContainerAwareInterface;

/**
 * Defines the BackgroundImageManagerInterface interface.
 */
interface BackgroundImageManagerInterface extends ContainerInjectionInterface {

  /**
   * Perform alterations before a form is rendered.
   *
   * @param array $form
   *   Nested array of form elements that comprise the form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form. The arguments that
   *   \Drupal::formBuilder()->getForm() was originally called with are
   *   available in the array $form_state->getBuildInfo()['args'].
   */
  public function alterEntityForm(array &$form, FormStateInterface $form_state);

  /**
   * Determines if a color is "dark".
   *
   * @param string $hex
   *   A HEX color representation.
   *
   * @return bool
   *   TRUE or FALSE
   */
  public function colorIsDark($hex = NULL);

  /**
   * Samples the average color of an image file.
   *
   * @param \Drupal\file\FileInterface $file
   *   A File entity object.
   * @param string $default
   *   The default lowercase simple color (HEX) representation to use if
   *   unable to sample the image.
   *
   * @return string
   *   The lowercase simple color (HEX) representation of the sampled image.
   */
  public function colorSampleFile(?FileInterface $file = NULL, $default = NULL);

  /**
   * Samples the average color of an image file.
   *
   * @param \Drupal\Core\Image\ImageInterface $image
   *   An Image object.
   * @param string $default
   *   The default lowercase simple color (HEX) representation to use if
   *   unable to sample the image.
   *
   * @return string
   *   The lowercase simple color (HEX) representation of the sampled image.
   */
  public function colorSampleImage(ImageInterface $image, $default = NULL);

  /**
   * Retrieves a background image based on the current route.
   *
   * @param string $langcode
   *   (optional) The language of the current context. Defaults to the current
   *   content language.
   * @param array $context
   *   (optional) An associative array of arbitrary data that can be useful to
   *   determine the proper fallback sequence.
   *
   * @return \Drupal\background_image\BackgroundImageInterface|null
   *   A background image entity, if one exists.
   */
  public function getBackgroundImage($langcode = NULL, array $context = []);

  /**
   * Retrieves the base class.
   *
   * @return string
   *   The base class.
   */
  public function getBaseClass();

  /**
   * Determines whether minified CSS URIs should be used.
   *
   * @return bool
   *   TRUE or FALSE
   */
  public function useMinifiedCssUri();

  /**
   * Builds the render array for the provided entity.
   *
   * @param \Drupal\background_image\BackgroundImageInterface $background_image
   *   The background image entity to render.
   * @param string $view_mode
   *   (optional) The view mode that should be used to render the entity.
   * @param string $langcode
   *   (optional) For which language the entity should be rendered, defaults to
   *   the current content language.
   *
   * @return array
   *   A render array for the entity.
   *
   * @throws \InvalidArgumentException
   *   Can be thrown when the set of parameters is inconsistent, like when
   *   trying to view a Comment and passing a Node which is not the one the
   *   comment belongs to, or not passing one, and having the comment node not
   *   be available for loading.
   */
  public function view($background_image, $view_mode = 'full', $langcode = NULL);

}
