<?php

namespace Drupal\background_image\Plugin\ContextReaction;

use Drupal\Component\Plugin\DependentPluginInterface;
use Drupal\Component\Uuid\UuidInterface;
use Drupal\context\ContextInterface;
use Drupal\context\ContextReactionPluginBase;
use Drupal\context\Form\AjaxFormTrait;
use Drupal\Core\Cache\Cache;
use Drupal\Core\Cache\CacheableDependencyInterface;
use Drupal\Core\Cache\CacheTagsInvalidatorInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Plugin\Context\ContextHandlerInterface;
use Drupal\Core\Plugin\Context\ContextRepositoryInterface;
use Drupal\Core\Plugin\PluginDependencyTrait;
use Drupal\Core\Session\AccountInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a content reaction.
 *
 * Set the page background as a reaction to context.
 *
 * @ContextReaction(
 *   id = "background_image",
 *   label = @Translation("Background Image")
 * )
 */
class BackgroundImage extends ContextReactionPluginBase implements ContainerFactoryPluginInterface, DependentPluginInterface, CacheableDependencyInterface {

  use AjaxFormTrait;

  use PluginDependencyTrait {
    addDependency as addDependencyTrait;
  }

  /**
   * The background image to be displayed with this reaction.
   *
   * @var array
   */
  protected $background_image = NULL;

  /**
   * {@inheritdoc}
   */
  public function __construct(
    array $configuration,
    $pluginId,
    $pluginDefinition,
    protected UuidInterface $uuid,
    protected ContextRepositoryInterface $contextRepository,
    protected ContextHandlerInterface $contextHandler,
    protected AccountInterface $account,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected CacheTagsInvalidatorInterface $cacheTagsInvalidator,
  ) {
    parent::__construct($configuration, $pluginId, $pluginDefinition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $pluginId, $pluginDefinition) {
    return new static(
      $configuration,
      $pluginId,
      $pluginDefinition,
      $container->get('uuid'),
      $container->get('context.repository'),
      $container->get('context.handler'),
      $container->get('current_user'),
      $container->get('entity_type.manager'),
      $container->get('cache_tags.invalidator'),
    );
  }

  /**
   * Executes the plugin.
   *
   * @return \Drupal\media\Entity\Media
   *   Background image to be displayed.
   */
  public function execute() {
    /** @var \Drupal\media\Entity\Media $backgroundImage */
    return $this->configuration['background_image'] ?
      $this->entityTypeManager->getStorage('media')
        ->load($this->configuration['background_image']) : NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
        'background_image' => NULL,
      ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function setConfiguration(array $configuration) {
    $this->configuration = $configuration + $this->defaultConfiguration();

    if (isset($configuration['background_image'])) {
      $this->background_image = $configuration['background_image'];
    }

    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getConfiguration() {
    return [
        'background_image' => $this->background_image,
      ] + parent::getConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function summary() {
    return $this->t('Lets you modify a background image when a context applies');
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state, ?ContextInterface $context = NULL) {
    $entities = $this->entityTypeManager->getStorage('media')
      ->loadByProperties(['bundle' => 'background_image']);

    $options = [NULL => '- None -'];
    foreach ($entities as $entity) {
      $options[$entity->id()] = $entity->label();
    }

    $form['background_image'] = [
      '#type' => 'select',
      '#title' => $this->t('Background image'),
      '#default_value' => $this->configuration['background_image'],
      '#description' => 'The background image media item to display.',
      '#options' => $options,
    ];
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    $config = $this->getConfiguration();
    $config['background_image'] = $form_state->getValue(['background_image'], NULL);
    $this->setConfiguration($config);
    $this->cacheTagsInvalidator->invalidateTags(['context.reaction.background']);
  }

  public function getCacheContexts() {
    return ['url.path'];
  }

  public function getCacheTags() {
    return [];
  }

  public function getCacheMaxAge() {
    return Cache::PERMANENT;
  }

}
