(function (Drupal, drupalSettings, once) {
  'use strict';

  let orderRef;
  let stopPolling = false;
  let collecting = false;
  let statusInterval;
  let startTime;
  let configurationHash;

  // @TODO: Add docs, clean up.
  Drupal.behaviors.bankIDAuthenticate = {
    attach: function (context) {

      configurationHash = drupalSettings.bankid.configurationHash || '';
      once('bankid-dialog', '.bankid-container', document).forEach(async (element) => {
        stopPolling = false;
        startTime = new Date();

        const qrButton = document.getElementById('bankid-qr-code-button');
        if (qrButton) {
          // Toggle fullscreen mode for QR code container.
          qrButton.addEventListener('click', (e) => {
            if (document.fullscreenElement !== null && document.fullscreenElement === e.currentTarget.closest('.bankid-container')) {
              void document.exitFullscreen();
            } else {
              void e.currentTarget.closest('.bankid-container').requestFullscreen();
            }
          });
        }

        const resetTimerButton = document.getElementById('bankid-expire-button');
        if (resetTimerButton) {
          // Reset timer and update ui on click
          resetTimerButton.addEventListener('click', () => {
            startTime = new Date();
            const expired = !syncExpiration();
            if (expired) {
              showElementsForStage('done');
            } else {
              showElementsForStage('qr-code');
            }
          });
        }

        await start();

        document.querySelectorAll('.ui-dialog-titlebar-close').forEach(function (element) {
          element.addEventListener('click', function () {
            if(!stopPolling){
              cancel();
            }
          });
        });

      });

    },
  };

  function ajaxRequest(method, url, parameters) {
    if (typeof parameters === 'object' && parameters !== null) {
      const queryString = new URLSearchParams(parameters).toString();
      if (queryString.length > 0) {
        url += '?' + queryString;
      }
    }
    return new Promise((resolve, reject) => {
      Drupal.ajax({
        url: url,
        method: method,
        contentType: 'application/json; charset=utf-8',
        dataType: 'json',
        success: function (data) {
          resolve(data);
        },
        error: function (data) {
          reject(new Error('Request failed with status: ' + data.status));
        },
      }).execute();
    });

  }

  function triggerBankIDEvent(type, data) {
    const event = new CustomEvent('bankid', {
      detail: {
        type: type,
        configurationHash: configurationHash,
        orderRef: orderRef,
        data: data || {},
      },
    });
    document.dispatchEvent(event);
  }

  async function start() {
    if (stopPolling) {
      return;
    }
    try {
      const params = {};
      if (configurationHash) {
        params.configurationHash = configurationHash;
      }
      const data = await ajaxRequest('GET', drupalSettings.bankid.startUrl, params);
      if (data.status === 'failed') {
        setStatusMessage(data.message || data.details || '');
        stopPolling = true;
        showElementsForStage('done');
        triggerBankIDEvent('failed', data.data || {});
        return;
      }
      orderRef = data.orderRef;

      renderAppLink(data.autoStartToken);
      startStatusPolling();
    } catch (err) {
      stopPolling = true;
      showElementsForStage('done');
      setStatusMessage(Drupal.t('An error occurred while trying to use BankID. Please try again'));
    }
  }

  async function status() {
    if (typeof orderRef === 'undefined' || stopPolling || collecting) {
      return;
    }

    try {
      collecting = true;
      const data = await ajaxRequest('GET', getUrlPath(drupalSettings.bankid.statusUrl, orderRef), null);
      collecting = false;

      setStatusMessage(data.message || '');

      switch (data.status) {
        case 'pending':
          if (data.hintCode === 'userSign') {
            showElementsForStage('throbber');
          } else {
            const expired = !syncExpiration();
            if (expired) {
              showElementsForStage('done');
            } else {
              showElementsForStage('qr-code');
              renderQrCode(data.qrData);
            }
          }
          break;
        case 'complete':
          stopPolling = true;
          showElementsForStage('done');
          triggerBankIDEvent('complete', data.data || {});
          break;
        case 'failed':
          if (data.hintCode === 'startFailed') {
            await start();
            return;
          }
          stopPolling = true;
          showElementsForStage('done');
          triggerBankIDEvent('failed', data.data || {});
          break;
      }
    } catch (err) {
      stopPolling = true;
      showElementsForStage('done');
      setStatusMessage(Drupal.t('An error occurred while checking the status. Please try again'));
    }
  }

  async function cancel() {
    stopPolling = true;
    const data = await ajaxRequest('GET', getUrlPath(drupalSettings.bankid.cancelUrl, orderRef));
    triggerBankIDEvent('cancel', data);
  }

  function getUrlPath(fullPath, orderRef) {
    if (fullPath.includes('{orderRef}')) {
      return fullPath.replace('{orderRef}', orderRef);
    }
  }

  function startStatusPolling() {
    if (stopPolling || statusInterval) {
      return;
    }

    void status();
    statusInterval = setInterval(() => {
      if (stopPolling) {
        // Stop polling
        if (statusInterval) {
          clearInterval(statusInterval);
          statusInterval = null;
        }
        return;
      }
      void status();
    }, 1000);
  }

  function renderQrCode(authData) {
    const qrButton = document.getElementById('bankid-qr-code-button');
    if (qrButton) {
      qrButton.classList.remove('visually-hidden');
    }
    const img = document.getElementById('qrcode');
    if (!img) {
      return;
    }

    img.classList.remove('visually-hidden');
    new QRious({
      element: img,
      size: 256,
      mime: 'image/png',
      value: authData,
    });
  }

  function renderAppLink(autoStartToken) {
    var link = document.getElementById('bankid-app-link');
    if (link) {
      var appLink = document.createElement('a');
      appLink.textContent = Drupal.t('Open BankID app on this device');
      appLink.href = 'bankid:///?autostarttoken=' + autoStartToken + '&redirect=null';
      appLink.classList.add('button');
      link.innerHTML = '';
      link.appendChild(appLink);
    }
  }

  function setStatusMessage(message) {
    const bankidStatusMessageElement = document.getElementById('bankid-status-message');
    if (bankidStatusMessageElement) {
      bankidStatusMessageElement.textContent = message;
    }
  }

  /**
   * Method to hide and show the correct elements based on a predefined set of stages.
   *
   * OBS! Is not responsible for setting new data, only clearing/resetting and
   * hiding/showing elements.
   * @param stage
   *   The stage to show elements for. Can be one of: throbber, qr-code, done.
   */
  function showElementsForStage(stage) {
    switch (stage) {
      case 'throbber':
        showElement(document.getElementById('bankid-throbber'), true);
        showElement(document.getElementById('bankid-progress-container'), false);
        clearQRCode();
        clearAppLink();
        break;
      case 'qr-code':
        const timeRemaining = timeToExpire();
        if (timeRemaining > 30) {
          showElement(document.getElementById('bankid-throbber'), false);
          showElement(document.getElementById('bankid-progress-container'), true);
          showElement(document.getElementById('bankid-expire-button'), false);
          showElement(document.getElementById('bankid-progress-bar'), true);
        } else if (timeRemaining > 0) {
          showElement(document.getElementById('bankid-progress-container'), true);
          showElement(document.getElementById('bankid-throbber'), false);
          showElement(document.getElementById('bankid-expire-button'), true);
          showElement(document.getElementById('bankid-progress-bar'), false);
        } else {
          showElement(document.getElementById('bankid-throbber'), false);
          showElement(document.getElementById('bankid-progress-container'), false);
          clearQRCode();
          clearAppLink();
        }
        break;
      case 'done':
        showElement(document.getElementById('bankid-throbber'), false);
        showElement(document.getElementById('bankid-progress-container'), false);
        clearQRCode();
        clearAppLink();
        break;
    }
  }

  function showElement(element, visible = true) {
    if (!element) {
      return;
    }
    if (visible && element.classList.contains('visually-hidden')) {
      element.classList.remove('visually-hidden');
    } else if (!visible && !element.classList.contains('visually-hidden')) {
      element.classList.add('visually-hidden');
    }
  }

  function clearQRCode() {
    const qrButton = document.getElementById('bankid-qr-code-button');
    const img = document.getElementById('qrcode');
    if (img) {
      img.src = '';
    }
    if (qrButton) {
      qrButton.classList.add('visually-hidden');
    }
  }

  function clearAppLink() {
    var link = document.getElementById('bankid-app-link');
    if (link) {
      link.innerHTML = '';
    }
  }

  /**
   * Synchronizes the expiration time with element values.
   *
   * When expired, cancels the authentication process and displays a message.
   *
   * @returns {boolean}
   *   True if not expired, false if expired.
   */
  function syncExpiration() {
    const timeRemaining = timeToExpire();
    let text;
    if (timeRemaining >= 60) {
      const minutes = Math.round(timeRemaining / 60);
      text = Drupal.t('@minutes minutes remaining', { '@minutes': minutes });
    } else if (timeRemaining > 30) {
      const tens = Math.floor(timeRemaining / 10) * 10;
      text = Drupal.t('@seconds seconds remaining', { '@seconds': tens });
    } else {
      text = Drupal.t('@seconds seconds remaining', { '@seconds': timeRemaining });
    }

    if (timeRemaining <= 0) {
      cancel();
      setStatusMessage(Drupal.t('Authentication has timed out. Please try again.'));
      return false;
    }

    const remainingTimeElement = document.getElementById('bankid-time-remaining');
    if (remainingTimeElement) {
      remainingTimeElement.textContent = text;
    }
    const progressBar = document.getElementById('bankid-progress-bar');
    if (progressBar) {
      progressBar.value = Math.floor(timeRemaining / drupalSettings.bankid.expiryTime * 100);
    }

    return true;
  }

  /**
   * Calculates the time remaining until the Bankid should not be renewed.
   *
   * @returns {number}
   *   The number of seconds remaining until the Bankid expires.
   */
  function timeToExpire() {
    const now = new Date();
    return drupalSettings.bankid.expiryTime - Math.floor((now.getTime() - startTime.getTime()) / 1000);
  }
})(Drupal, drupalSettings, once);
