<?php

declare(strict_types=1);

namespace Drupal\bankid;

use Drupal\bankid\ValueObjects\BankIDOrderDoneData;

/**
 * Interface for BankID Case plugins.
 */
interface BankIDCaseInterface {

  /**
   * Returns the translated plugin label.
   */
  public function label(): string;

  /**
   * Set the private data.
   *
   * @param array $private_data
   *   The private data.
   */
  public function setPrivateData(array $private_data): void;

  /**
   * Get the requirements for the order.
   *
   * @return array
   *   An array of requirements for the order.
   *
   * @see https://developers.bankid.com/api-references/auth--sign/auth
   * @see https://developers.bankid.com/api-references/auth--sign/sign
   */
  public function getRequirements(): array;

  /**
   * Get the hidden data for the order.
   *
   * @return string
   *   The hidden data for the order.
   *
   * @see https://developers.bankid.com/api-references/auth--sign/auth
   * @see https://developers.bankid.com/api-references/auth--sign/sign
   */
  public function getHiddenData(): string;

  /**
   * Get the visible data for the order.
   *
   * @return string
   *   The visible data for the order.
   *
   * @see https://developers.bankid.com/api-references/auth--sign/auth
   * @see https://developers.bankid.com/api-references/auth--sign/sign
   */
  public function getVisibleData(): string;

  /**
   * Get the visible data format for the order.
   *
   * @return string
   *   The visible data format for the order, 'plaintext' or 'simpleMarkdown'.
   *
   * @see https://developers.bankid.com/api-references/auth--sign/auth
   * @see https://developers.bankid.com/api-references/auth--sign/sign
   */
  public function getVisibleDataFormat(): string;

  /**
   * Called when the authentication is complete.
   *
   * This method can be responsible, for example, for creating a user account
   * or logging in an existing user based on the response from the BankID
   * service.
   *
   * @param array $response
   *   The authentication complete response from BankID.
   *
   * @return \Drupal\bankid\ValueObjects\BankIDOrderDoneData
   *   Data that will be sent to the client for further steps, such as userid
   *   of authenticated user, or any other relevant information.
   */
  public function orderCompleted(array $response): BankIDOrderDoneData;

  /**
   * Called when the order is cancelled.
   *
   * @param array $response
   *   The response from the BankID service.
   * @param string $order_ref
   *   The order reference.
   *
   * @return \Drupal\bankid\ValueObjects\BankIDOrderDoneData
   *   Data that will be sent to the client for further steps, such as userid
   *   of authenticated user, or any other relevant information.
   */
  public function orderCancelled(array $response, string $order_ref): BankIDOrderDoneData;

  /**
   * Called when the order fails.
   *
   * @param array $response
   *   The response from the BankID service.
   *
   * @return \Drupal\bankid\ValueObjects\BankIDOrderDoneData
   *   Data that will be sent to the client for further steps, such as userid
   *   of authenticated user, or any other relevant information.
   */
  public function orderFailed(array $response): BankIDOrderDoneData;

}
