<?php

namespace Drupal\bankid;

use Drupal\bankid\Enum\OrderOperation;
use Drupal\Core\Config\ConfigFactory;
use Drupal\Core\File\FileSystemInterface;
use Drupal\key\KeyRepository;
use Drupal\key\Plugin\KeyPluginInterface;
use Drupal\key\Plugin\KeyProvider\FileKeyProvider;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\RequestException;
use GuzzleHttp\RequestOptions;

/**
 * Class handling BankID integration.
 *
 * @todo You can't extend a final class. See https://github.com/mailjet/mailjet-apiv3-php/commit/2405918440bcc51c6f5441960cae3e4a5ce8dc94
 *   for alternative solution.
 *
 * @package Drupal\bankid
 */
class BankIDClient extends Client implements BankIDClientInterface {

  /**
   * Constructs a new \Drupal\bankid\BankIDClient object.
   *
   * @param \Drupal\Core\Config\ConfigFactory $config_factory
   *   The config factory.
   * @param \Drupal\key\KeyRepository $key_repository
   *   The key repository.
   * @param \Drupal\Core\File\FileSystemInterface $file_system
   *   The file system.
   */
  public function __construct(ConfigFactory $config_factory, KeyRepository $key_repository, FileSystemInterface $file_system) {
    $config = $config_factory->get('bankid.settings')->get($config_factory->get('bankid.settings')->get('environment'));
    /** @var \Drupal\key\Plugin\KeyPluginInterface $rp_provider */
    $rp_provider = $key_repository->getKey($config['rp_certificate'])->getKeyProvider();
    /** @var \Drupal\key\Plugin\KeyPluginInterface $srv_cert_provider */
    $srv_cert_provider = $key_repository->getKey($config['issuer_of_server_certificate'])->getKeyProvider();
    if (!$rp_provider == $this->isKeyFileProvider($rp_provider) || !$srv_cert_provider == $this->isKeyFileProvider($srv_cert_provider)) {
      throw new \InvalidArgumentException(
        'The BankID client requires the rp_certificate and issuer_of_server_certificate keys to be provided by a FileKeyProvider.'
      );
    }
    $httpOptions = [
      'base_uri' => $config['api_base_url'] . '/',
      'cert' => [
        $file_system->realpath($rp_provider->getConfiguration()['file_location']),
        $key_repository->getKey($config['rp_passphrase'])->getKeyValue(),
      ],
      'verify' => $file_system->realpath($srv_cert_provider->getConfiguration()['file_location']),
      'headers' => [
        'Content-Type' => 'application/json',
        'Accept' => 'application/json',
      ],
    ];

    parent::__construct($httpOptions);
  }

  /**
   * Check if a key provider is a FileKeyProvider.
   *
   * @param \Drupal\key\Plugin\KeyPluginInterface $key_provider
   *   The key provider.
   *
   * @return bool
   *   TRUE if the key provider is a FileKeyProvider. Else FALSE.
   */
  private function isKeyFileProvider(KeyPluginInterface $key_provider) {
    return $key_provider instanceof FileKeyProvider && $key_provider->getPluginId() === 'file';
  }

  /**
   * {@inheritDoc}
   */
  public function start(
    OrderOperation $operation,
    $endUserIp = '127.0.0.1',
    $requirement = NULL,
    $userVisibleData = NULL,
    $userNonVisibleData = NULL,
    $userVisibleDataFormat = NULL,
  ): BankIDResponse {
    $payload['endUserIp'] = $endUserIp;

    if (!empty($requirement)) {
      $payload['requirement'] = $requirement;
    }

    if (!empty($userVisibleData)) {
      $payload['userVisibleData'] = base64_encode($userVisibleData);
    }

    if (!empty($userNonVisibleData)) {
      $payload['userNonVisibleData'] = base64_encode($userNonVisibleData);
    }

    if (!empty($userVisibleDataFormat)) {
      $payload['userVisibleDataFormat'] = $userVisibleDataFormat;
    }

    try {
      $httpResponse = $this->post($operation->value, [
        RequestOptions::JSON => $payload,
      ]);
    }
    catch (RequestException $e) {
      return $this->requestExceptionToBankIdResponse($e);
    }

    $httpResponseBody = json_decode($httpResponse->getBody(), TRUE);

    return new BankIDResponse(BankIDResponse::STATUS_PENDING, $httpResponseBody);
  }

  /**
   * {@inheritDoc}
   */
  public function phoneAuth($personalNumber, $callInitiator, $requirement = NULL, $userVisibleData = NULL, $userNonVisibleData = NULL, $userVisibleDataFormat = NULL): BankIDResponse {
    try {
      $payload = [
        'personalNumber' => $personalNumber,
        'callInitiator' => $callInitiator,
      ];

      if (!empty($requirement)) {
        $payload['requirement'] = $requirement;
      }

      if (!empty($userVisibleData)) {
        $payload['userVisibleData'] = base64_encode($userVisibleData);
      }

      if (!empty($userNonVisibleData)) {
        $payload['userNonVisibleData'] = base64_encode($userNonVisibleData);
      }

      if (!empty($userVisibleDataFormat)) {
        $payload['userVisibleDataFormat'] = $userVisibleDataFormat;
      }

      $httpResponse = $this->post('phone/auth', [
        RequestOptions::JSON => $payload,
      ]);
    }
    catch (RequestException $e) {
      return $this->requestExceptionToBankIdResponse($e);
    }

    $httpResponseBody = json_decode($httpResponse->getBody(), TRUE);

    return new BankIDResponse(BankIDResponse::STATUS_PENDING, $httpResponseBody);
  }

  /**
   * {@inheritDoc}
   */
  public function phoneSign($personalNumber, $callInitiator, $requirement = NULL, $userVisibleData = NULL, $userNonVisibleData = NULL, $userVisibleDataFormat = NULL): BankIDResponse {
    try {
      $payload = [
        'personalNumber' => $personalNumber,
        'callInitiator' => $callInitiator,
      ];

      if (!empty($requirement)) {
        $payload['requirement'] = $requirement;
      }

      if (!empty($userVisibleData)) {
        $payload['userVisibleData'] = base64_encode($userVisibleData);
      }

      if (!empty($userNonVisibleData)) {
        $payload['userNonVisibleData'] = base64_encode($userNonVisibleData);
      }

      if (!empty($userVisibleDataFormat)) {
        $payload['userVisibleDataFormat'] = $userVisibleDataFormat;
      }

      $httpResponse = $this->post('phone/sign', [
        RequestOptions::JSON => $payload,
      ]);
    }
    catch (RequestException $e) {
      return $this->requestExceptionToBankIdResponse($e);
    }

    $httpResponseBody = json_decode($httpResponse->getBody(), TRUE);

    return new BankIDResponse(BankIDResponse::STATUS_PENDING, $httpResponseBody);
  }

  /**
   * {@inheritDoc}
   */
  public function collect($orderReference): BankIDResponse {
    try {
      $httpResponse = $this->post('collect', [
        RequestOptions::JSON => [
          'orderRef' => $orderReference,
        ],
      ]);
    }
    catch (RequestException $e) {
      return $this->requestExceptionToBankIdResponse($e);
    }

    $httpResponseBody = json_decode($httpResponse->getBody(), TRUE);
    return new BankIDResponse($httpResponseBody['status'], $httpResponseBody);
  }

  /**
   * {@inheritDoc}
   */
  public function cancel($orderReference): BankIDResponse {
    try {
      $httpResponse = $this->post('cancel', [
        RequestOptions::JSON => [
          'orderRef' => $orderReference,
        ],
      ]);
    }
    catch (RequestException $e) {
      return $this->requestExceptionToBankIdResponse($e);
    }

    $httpResponseBody = json_decode($httpResponse->getBody(), TRUE);

    return new BankIDResponse(BankIDResponse::STATUS_OK, $httpResponseBody);
  }

  /**
   * Transform GuzzleHttp request exception into a BankIDResponse.
   *
   * @param \GuzzleHttp\Exception\RequestException $e
   *   The request exception.
   *
   * @return BankIDResponse
   *   The BankID response.
   */
  private function requestExceptionToBankIdResponse(RequestException $e): BankIDResponse {
    $body = $e->hasResponse() ? $e->getResponse()->getBody() : NULL;

    if ($body) {
      return new BankIDResponse(BankIDResponse::STATUS_FAILED, json_decode($body, TRUE));
    }

    return new BankIDResponse(BankIDResponse::STATUS_FAILED, ['errorMessage' => $e->getMessage()]);
  }

}
