<?php

namespace Drupal\bankid;

use Drupal\bankid\Enum\OrderOperation;

interface BankIDClientInterface {

  /**
   * Authenticate a user using animated QR code.
   *
   * @param \Drupal\bankid\Enum\OrderOperation $operation
   *   BankID operation (endpoint).
   * @param string $endUserIp
   *   The end user IP.
   * @param string $requirement
   *   The requirement.
   * @param string $userVisibleData
   *   The user visible data.
   * @param string $userNonVisibleData
   *   The user non-visible data.
   * @param string $userVisibleDataFormat
   *   The user visible data format.
   *
   * @return \Drupal\bankid\BankIDResponse
   *   The BankID response.
   *
   * @link https://www.bankid.com/utvecklare/guider/teknisk-integrationsguide/graenssnittsbeskrivning/auth
   */
  public function start(OrderOperation $operation, $endUserIp = '127.0.0.1', $requirement = NULL, $userVisibleData = NULL, $userNonVisibleData = NULL, $userVisibleDataFormat = NULL): BankIDResponse;

  /**
   * Authenticate a user over phone.
   *
   * @param string $personalNumber
   *   The personal number.
   * @param string $callInitiator
   *   The call initiator.
   * @param string $requirement
   *   The requirement.
   * @param string $userVisibleData
   *   The user visible data.
   * @param string $userNonVisibleData
   *   The user non visible data.
   * @param string $userVisibleDataFormat
   *   The user visible data format.
   *
   * @return BankIDResponse
   *   The BankID response.
   *
   * @link https://www.bankid.com/utvecklare/guider/teknisk-integrationsguide/graenssnittsbeskrivning/phone-auth
   */
  public function phoneAuth($personalNumber, $callInitiator, $requirement = NULL, $userVisibleData = NULL, $userNonVisibleData = NULL, $userVisibleDataFormat = NULL): BankIDResponse;

  /**
   * Request a signing order for a user over the phone.
   *
   * @param string $personalNumber
   *   The personal number.
   * @param string $callInitiator
   *   The call initiator.
   * @param string $requirement
   *   The requirement.
   * @param string $userVisibleData
   *   The user visible data.
   * @param string $userNonVisibleData
   *   The user non visible data.
   * @param string $userVisibleDataFormat
   *   The user visible data format.
   *
   * @return BankIDResponse
   *   The BankID response.
   *
   * @link https://www.bankid.com/utvecklare/guider/teknisk-integrationsguide/graenssnittsbeskrivning/phone-sign
   */
  public function phoneSign($personalNumber, $callInitiator, $requirement = NULL, $userVisibleData = NULL, $userNonVisibleData = NULL, $userVisibleDataFormat = NULL): BankIDResponse;

  /**
   * Collect an ongoing user request.
   *
   * @param string $orderReference
   *   The order reference.
   *
   * @return BankIDResponse
   *   The BankID response.
   *
   * @link https://www.bankid.com/utvecklare/guider/teknisk-integrationsguide/graenssnittsbeskrivning/collect
   */
  public function collect($orderReference): BankIDResponse;

  /**
   * Cancel an ongoing order per the users request.
   *
   * @param string $orderReference
   *   The order reference.
   *
   * @return BankIDResponse
   *   The BankID response.
   *
   * @link https://www.bankid.com/utvecklare/guider/teknisk-integrationsguide/graenssnittsbeskrivning/cancel
   */
  public function cancel($orderReference): BankIDResponse;
}
