<?php

declare(strict_types=1);

namespace Drupal\bankid;

use Drupal\bankid\Enum\OrderOperation;
use Drupal\bankid\ValueObjects\CaseData;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\DependencyInjection\DependencySerializationTrait;
use Drupal\Core\Routing\RouteProviderInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\Core\Url;

/**
 * Builds UI elements for BankID cases.
 */
class BankIDManager {

  use StringTranslationTrait;
  use DependencySerializationTrait;

  /**
   * The BankID configuration.
   */
  protected ImmutableConfig $config;

  /**
   * Constructs a BankIDManager object.
   */
  public function __construct(
    private readonly RouteProviderInterface $routeProvider,
    TranslationInterface $stringTranslation,
    ConfigFactoryInterface $config_factory,
    protected BankIDOrderManager $orderManager,
  ) {
    $this->stringTranslation = $stringTranslation;
    $this->config = $config_factory->get('bankid.settings');
  }

  /**
   * Builds the dialog for BankID authentication.
   *
   * @param string|null $configuration_hash
   *   The configuration hash to identify the plugin configuration for this
   *   bankid order.
   *
   * @return array
   *   The renderable array.
   */
  public function buildDialog(?string $configuration_hash = NULL): array {
    $build = [
      '#type' => 'container',
      'bankid_qr' => [
        '#theme' => 'bankid_dialog',
        '#attached' => [
          'library' => ['bankid/bankid.client'],
          'drupalSettings' => [
            'bankid' => [
              'configurationHash' => $configuration_hash,
              'startUrl' => $this->routeProvider
                ->getRouteByName('bankid.start')->getPath(),
              'statusUrl' => $this->routeProvider
                ->getRouteByName('bankid.status')->getPath(),
              'cancelUrl' => $this->routeProvider
                ->getRouteByName('bankid.cancel')->getPath(),
              'expiryTime' => $this->config->get('expiry_time') ?? 300,
            ],
          ],
        ],
      ],
    ];

    if ($this->isMockMode()) {
      $build['mock'] = \Drupal::formBuilder()
        ->getForm('\Drupal\bankid\Form\BankIDMockForm');
    }
    return $build;
  }

  private function isMockMode(): bool {
    $environment = $this->config->get('environment');
    return $environment === 'mock';
  }

  /**
   * Builds a link to open the BankID dialog.
   *
   * @param \Drupal\bankid\ValueObjects\CaseData $case
   *   The case data.
   *
   * @return array
   *   The renderable array.
   */
  public function buildDialogLink(CaseData $case): array {
    $configuration_hash = $this->orderManager->registerCase($case);

    return [
      '#type' => 'link',
      '#title' => $case->operation === OrderOperation::Sign ? $this->t('Sign with Mobile BankID') : $this->t('Authenticate with Mobile BankID'),
      '#url' => Url::fromRoute('bankid.dialog', [
        'nojs' => 'nojs',
        'configuration_hash' => $configuration_hash,
      ]),
      '#attributes' => [
        'class' => ['use-ajax', 'bankid-dialog-link'],
      ],
      '#cache' => [
        'max-age' => 0,
        'contexts' => ['session', 'user'],
      ],
    ];
  }

  /**
   * Builds an ajax button to open the BankID dialog.
   *
   * @param \Drupal\bankid\ValueObjects\CaseData $case
   *   The case data.
   *
   * @return array
   *   The renderable array.
   */
  public function buildDialogAjaxButton(CaseData $case): array {
    $configuration_hash = $this->orderManager->registerCase($case);

    return [
      '#type' => 'button',
      '#configuration_hash' => $configuration_hash,
      '#value' => $case->operation === OrderOperation::Sign ? $this->t('Sign with Mobile BankID') : $this->t('Authenticate with Mobile BankID'),
      '#ajax' => [
        'url' => Url::fromRoute('bankid.dialog', [
          'nojs' => 'nojs',
          'configuration_hash' => $configuration_hash,
        ]),
        'event' => 'click',
      ],
      // @todo refine caching based on case data (private data or not).
      '#cache' => [
        'max-age' => 0,
        'contexts' => ['session', 'user'],
      ],
    ];
  }

  /**
   * Get the order uid from the order ref.
   *
   * @param string $order_ref
   *   The order reference.
   *
   * @return string|null
   *   The order uid or NULL.
   */
  public function getOrderUid(string $order_ref): ?string {
    return $this->orderManager->getOrderUserId($order_ref);
  }

  /**
   * Get the order done temporary data.
   *
   * @param string $order_ref
   *   The order reference.
   *
   * @return array|null
   *   The temporary order data or NULL.
   */
  public function getOrderDoneTempData($order_ref): ?array {
    return $this->orderManager->getOrderDoneTempData($order_ref);
  }

}
